/*
 * Copyright © 2020-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of gaming-gui.
 *
 * Gaming-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Gaming-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * gaming-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.gui;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import de.fhdw.gaming.core.domain.DefaultGameBuilderFactoryProvider;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.util.GameBuilderFactoryWrapper;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.gui.util.FXUtil;
import de.fhdw.gaming.gui.util.FXUtil.ButtonDescriptor;
import javafx.application.Application;
import javafx.beans.binding.DoubleBinding;
import javafx.beans.property.SimpleBooleanProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ObservableBooleanValue;
import javafx.collections.ListChangeListener;
import javafx.event.ActionEvent;
import javafx.geometry.Insets;
import javafx.geometry.Pos;
import javafx.geometry.Rectangle2D;
import javafx.scene.Node;
import javafx.scene.Scene;
import javafx.scene.control.Alert.AlertType;
import javafx.scene.control.Button;
import javafx.scene.control.ListView;
import javafx.scene.layout.Border;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.BorderStroke;
import javafx.scene.layout.BorderStrokeStyle;
import javafx.scene.layout.CornerRadii;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Priority;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;
import javafx.stage.Screen;
import javafx.stage.Stage;
import javafx.stage.WindowEvent;

/**
 * Provides the main entry point.
 */
public final class Main extends Application {

    /**
     * The application title.
     */
    private static final String APP_NAME = "FHDW Gaming GUI";
    /**
     * The golden ratio between the width and the height of a "visually appealing" rectangle.
     */
    private static final double GOLDEN_RATIO = 1.618d;
    /**
     * Type of game.
     */
    private static final String PARAM_GAME = "game";

    /**
     * The registered GUI observers.
     */
    private List<GuiObserver> observers;
    /**
     * The root pane.
     */
    private BorderPane rootPane;
    /**
     * The log.
     */
    private ListView<String> log;
    /**
     * The pane with buttons on the start page.
     */
    private HBox startPageButtonPane;
    /**
     * The pane with buttons on the game page.
     */
    private VBox gamePageButtonPane;
    /**
     * The {@link GameRunner} when a game is running.
     */
    private SimpleObjectProperty<GameRunner> gameRunner;
    /**
     * Property which indicates if a game is running.
     */
    private SimpleBooleanProperty gameRunning;
    /**
     * Property which indicates if a game is finished.
     */
    private SimpleBooleanProperty gameFinished;
    /**
     * The last {@link GameBuilderFactory} used.
     */
    private Optional<GameBuilderFactory> lastGameBuilderFactory;
    /**
     * The last {@link GameBuilder} used.
     */
    private Optional<GameBuilder> lastGameBuilder;
    /**
     * The next available game identifier.
     */
    private int nextGameId;

    @Override
    public void start(final Stage primaryStage) {
        this.observers = Collections.synchronizedList(
                GuiObserverFactory.getInstances().stream().map(GuiObserverFactory::createObserver)
                        .collect(Collectors.toUnmodifiableList()));
        this.gameRunner = new SimpleObjectProperty<>();
        this.gameRunning = new SimpleBooleanProperty(false);
        this.gameFinished = new SimpleBooleanProperty(false);

        this.lastGameBuilderFactory = Optional.empty();
        this.lastGameBuilder = Optional.empty();
        this.nextGameId = 1;

        this.createUi(primaryStage);
    }

    /**
     * Creates the user interface.
     *
     * @param stage The primary stage.
     */
    private void createUi(final Stage stage) {
        Application.setUserAgentStylesheet(Application.STYLESHEET_MODENA);

        final Screen screen = Screen.getPrimary();
        final Rectangle2D bounds = screen.getVisualBounds();
        final double height = bounds.getHeight() * 3 / 5;
        final double width = Math.min(bounds.getWidth(), height * Main.GOLDEN_RATIO);

        this.rootPane = new BorderPane();

        this.createStartPageButtons(stage);
        this.createGamePageButtons(stage);
        this.rootPane.setCenter(this.startPageButtonPane);

        final Node logPane = this.createLogPane();
        this.rootPane.setBottom(logPane);

        final Scene scene = new Scene(this.rootPane, width, height);
        stage.setTitle(Main.APP_NAME);
        stage.setScene(scene);
        stage.sizeToScene();
        stage.show();

        stage.setMinWidth(stage.getWidth());
        stage.setMinHeight(stage.getHeight());
        stage.setMaximized(true);

        stage.setOnCloseRequest((final WindowEvent event) -> {
            this.stopGame();
            this.leaveGame();
        });
    }

    /**
     * Creates the buttons on the start page.
     *
     * @param stage The primary stage.
     */
    private void createStartPageButtons(final Stage stage) {
        final Button newGameButton = createNewGameButton();
        newGameButton.setOnAction((final ActionEvent event) -> this.startNewGame(stage));

        final Button exitButton = createExitButton();
        exitButton.setOnAction((final ActionEvent event) -> stage.close());

        final DoubleBinding prefWidthProperty = new DoubleBinding() {
            {
                this.bind(newGameButton.heightProperty());
                this.bind(exitButton.heightProperty());
            }

            @Override
            protected double computeValue() {
                return Math.max(newGameButton.getHeight(), exitButton.getHeight());
            }
        };

        final DoubleBinding prefHeightProperty = new DoubleBinding() {
            {
                this.bind(newGameButton.widthProperty());
                this.bind(exitButton.widthProperty());
            }

            @Override
            protected double computeValue() {
                return Math.max(newGameButton.getWidth(), exitButton.getWidth());
            }
        };

        newGameButton.prefWidthProperty().bind(prefWidthProperty);
        exitButton.prefWidthProperty().bind(prefWidthProperty);

        newGameButton.prefHeightProperty().bind(prefHeightProperty);
        exitButton.prefHeightProperty().bind(prefHeightProperty);

        this.startPageButtonPane = new HBox();
        this.startPageButtonPane.getChildren().addAll(newGameButton, exitButton);
        HBox.setHgrow(newGameButton, Priority.ALWAYS);
        HBox.setHgrow(exitButton, Priority.ALWAYS);
    }

    /**
     * Creates the buttons on the game page.
     *
     * @param stage The primary stage.
     */
    private void createGamePageButtons(final Stage stage) {
        final Button startOrPauseGameButton = createContinueOrPauseGameButton(this.gameRunning);
        final Button stopGameButton = createStopGameButton();
        final Button restartGameButton = createRestartGameButton();
        final Button leaveGameButton = createLeaveGameButton();

        final DoubleBinding prefWidthProperty = new DoubleBinding() {
            {
                this.bind(startOrPauseGameButton.heightProperty());
                this.bind(stopGameButton.heightProperty());
                this.bind(restartGameButton.heightProperty());
                this.bind(leaveGameButton.heightProperty());
            }

            @Override
            protected double computeValue() {
                return Stream
                        .of(startOrPauseGameButton.getHeight(), stopGameButton.getHeight(),
                                restartGameButton.getHeight(), leaveGameButton.getHeight())
                        .max(Double::compareTo).orElseThrow();
            }
        };

        startOrPauseGameButton.prefWidthProperty().bind(prefWidthProperty);
        stopGameButton.prefWidthProperty().bind(prefWidthProperty);
        restartGameButton.prefWidthProperty().bind(prefWidthProperty);
        leaveGameButton.prefWidthProperty().bind(prefWidthProperty);

        startOrPauseGameButton.disableProperty().bind(this.gameRunner.isNull().or(this.gameFinished));
        stopGameButton.disableProperty().bind(this.gameRunner.isNull().or(this.gameFinished));
        restartGameButton.disableProperty().bind(this.gameRunner.isNull().or(this.gameFinished.not()));
        leaveGameButton.disableProperty().bind(this.gameRunner.isNull().or(this.gameFinished.not()));

        this.gamePageButtonPane = new VBox();
        this.gamePageButtonPane.getChildren().addAll(startOrPauseGameButton, stopGameButton, restartGameButton,
                leaveGameButton);
        this.gamePageButtonPane.setFillWidth(false);
        VBox.setVgrow(startOrPauseGameButton, Priority.ALWAYS);
        VBox.setVgrow(stopGameButton, Priority.ALWAYS);
        VBox.setVgrow(restartGameButton, Priority.ALWAYS);
        VBox.setVgrow(leaveGameButton, Priority.ALWAYS);

        startOrPauseGameButton.setOnAction((final ActionEvent event) -> {
            if (this.gameRunning.get()) {
                this.pauseGame();
                this.log.getItems().add("Game paused.");
            } else {
                this.continueGame();
                this.log.getItems().add("Game continued.");
            }
        });
        stopGameButton.setOnAction((final ActionEvent event) -> {
            this.stopGame();
            this.log.getItems().add("Game stopped.");
        });
        restartGameButton.setOnAction((final ActionEvent event) -> {
            this.stopGame();
            this.restartGame(stage);
            this.log.getItems().add("Game restarted.");
        });
        leaveGameButton.setOnAction((final ActionEvent event) -> {
            this.stopGame();
            this.leaveGame();
            this.log.getItems().add("Game left.");
        });
    }

    /**
     * Returns the "start-or-pause game" button.
     *
     * @param mode Switches between the start-or-continue mode ({@code false}) or the pause mode ({@code true}).
     */
    private static Button createContinueOrPauseGameButton(final ObservableBooleanValue mode) {
        return FXUtil.createTwoModeButtonWithSVGImage(
                mode,
                new ButtonDescriptor().text("Continue").description("Continues the game.").icon("media-playback-start"),
                new ButtonDescriptor().text("Pause").description("Pauses the game.").icon("media-playback-pause"));
    }

    /**
     * Returns the "stop game" button.
     */
    private static Button createStopGameButton() {
        return FXUtil.createButtonWithSVGImage(
                new ButtonDescriptor().text("Stop").description("Stops the game.").icon("media-playback-stop"));
    }

    /**
     * Returns the "restart game" button.
     */
    private static Button createRestartGameButton() {
        return FXUtil.createButtonWithSVGImage(
                new ButtonDescriptor().text("Restart").description("Restarts the game.").icon("media-skip-backward"));
    }

    /**
     * Returns the "leave game" button.
     */
    private static Button createLeaveGameButton() {
        return FXUtil.createButtonWithSVGImage(
                new ButtonDescriptor().text("Leave").description("Leaves the game.").icon("media-eject"));
    }

    /**
     * Returns the "new game" button.
     */
    private static Button createNewGameButton() {
        return FXUtil.createButtonWithSVGImage(
                new ButtonDescriptor().text("New").description("Starts a new game.").icon("media-mount"));
    }

    /**
     * Returns the "exit" button.
     */
    private static Button createExitButton() {
        return FXUtil.createButtonWithSVGImage(
                new ButtonDescriptor().text("Exit").description("Exits application.").icon("system-shutdown"));
    }

    /**
     * Creates and returns the log.
     */
    private Node createLogPane() {
        this.log = new ListView<>();
        this.log.setEditable(false);
        this.log.setPrefHeight(200.0);
        this.log.getItems().addListener((final ListChangeListener.Change<? extends String> c) ->
            this.log.scrollTo(this.log.getItems().size() - 1));

        final VBox vBox = new VBox(this.log);
        vBox.setPadding(new Insets(5.0d));
        vBox.setAlignment(Pos.BOTTOM_CENTER);
        vBox.setBorder(
                new Border(
                        new BorderStroke(
                                Color.BLACK,
                                BorderStrokeStyle.SOLID,
                                CornerRadii.EMPTY,
                                BorderStroke.DEFAULT_WIDTHS)));
        VBox.setVgrow(this.log, Priority.NEVER);

        LogObserver.INSTANCE.setLog(this.log);
        return vBox;
    }

    /**
     * Starts a new game.
     *
     * @param stage The primary stage.
     */
    private void startNewGame(final Stage stage) {
        final GridPane gamePane = new GridPane();
        gamePane.setPrefSize(Double.MAX_VALUE, Double.MAX_VALUE);
        this.rootPane.setCenter(gamePane);
        this.rootPane.setRight(this.gamePageButtonPane);

        try {
            final GameBuilder gameBuilder;
            if (this.lastGameBuilder.isPresent()) {
                gameBuilder = this.lastGameBuilder.orElseThrow();
            } else {
                final GameBuilderFactory gameBuilderFactory = this.determineGameBuilderFactory(stage);
                gameBuilder = gameBuilderFactory.createGameBuilder(
                        gameBuilderFactory.extendInputProvider(new InteractiveDialogInputProvider(stage)));
                this.lastGameBuilder = Optional.of(gameBuilder);
            }

            this.gameRunner.set(new GameRunner(this.observers, gameBuilder.build(this.nextGameId++), gamePane));
            this.gameRunning.bind(this.gameRunner.get().runningProperty());
            this.gameFinished.bind(this.gameRunner.get().finishedProperty());

            this.gameRunner.get().startGame();
            this.gamePageButtonPane.getChildren().get(0).requestFocus();
        } catch (final Exception e) {
            FXUtil.showAlert(stage, AlertType.ERROR, e.getMessage());
            this.leaveGame();
        }
    }

    /**
     * Continues the game.
     */
    private void continueGame() {
        this.gameRunner.get().continueGame();
    }

    /**
     * Pauses the game.
     */
    private void pauseGame() {
        this.gameRunner.get().pauseGame();
    }

    /**
     * Stops the game if necessary.
     */
    private void stopGame() {
        if (this.gameRunner.get() != null) {
            this.gameRunner.get().abortGame();
        }
    }

    /**
     * Restarts the game.
     *
     * @param stage The primary stage.
     */
    private void restartGame(final Stage stage) {
        if (this.gameRunner.get() != null) {
            this.gameRunner.get().abortGame();
            this.gameRunner.get().terminate();
        }

        this.gameRunner.set(null);
        this.gameRunning.unbind();
        this.gameFinished.unbind();
        this.gameRunning.set(false);
        this.gameFinished.set(false);

        this.startNewGame(stage);
    }

    /**
     * Leaves the game. Requires that no game is running.
     */
    private void leaveGame() {
        if (this.gameRunner.get() != null) {
            this.gameRunner.get().terminate();
        }

        this.gameRunner.set(null);
        this.gameRunning.unbind();
        this.gameFinished.unbind();
        this.gameRunning.set(false);
        this.gameFinished.set(false);

        this.lastGameBuilder = Optional.empty();
        this.lastGameBuilderFactory = Optional.empty();

        this.rootPane.setCenter(this.startPageButtonPane);
        this.rootPane.setRight(null);
        this.startPageButtonPane.getChildren().get(0).requestFocus();
    }

    /**
     * Determines the {@link GameBuilderFactory} to use.
     *
     * @param stage The primary stage.
     */
    private GameBuilderFactory determineGameBuilderFactory(final Stage stage) throws InputProviderException {

        if (this.lastGameBuilderFactory.isPresent()) {
            return this.lastGameBuilderFactory.orElseThrow();
        }

        final Set<GameBuilderFactory> gameBuilderFactories = new LinkedHashSet<>();
        for (final GameBuilderFactory gameBuilderFactory : new DefaultGameBuilderFactoryProvider()
                .getGameBuilderFactories()) {
            gameBuilderFactories.add(new GameBuilderFactoryWrapper(gameBuilderFactory));
        }

        final InputProvider inputProvider = new InteractiveDialogInputProvider(stage);
        inputProvider.needObject(Main.PARAM_GAME, "Which game to play", Optional.empty(), gameBuilderFactories);
        final Map<String, Object> data = inputProvider.requestData("Competition parameters");

        this.lastGameBuilderFactory = Optional.of((GameBuilderFactory) data.get(Main.PARAM_GAME));
        return this.lastGameBuilderFactory.orElseThrow();
    }

    /**
     * The main entry point.
     *
     * @param parameters The parameters passed to the application.
     */
    public static void main(final String[] parameters) {
        // the whole application is English-only, so use English also for JavaFX messages
        Locale.setDefault(Locale.ENGLISH);
        launch(Main.class, parameters);
    }
}
