package de.fhdw.gaming.ipspiel21.searchtrees.domain.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.State;
import de.fhdw.gaming.ipspiel21.searchtrees.domain.Evaluation;
import de.fhdw.gaming.ipspiel21.searchtrees.domain.NextMoveGenerator;
import de.fhdw.gaming.ipspiel21.searchtrees.domain.SearchTreeAlgorithms;

/**
 *
 * Implements {@link SearchTreeAlgorithms} by implementation of the Minimax algorithm.
 *
 * @param <P>
 * @param <S>
 * @param <M>
 */
public class MinimaxImpl<P extends Player, S extends State<P, S>, M extends Move<P, S>>
        implements SearchTreeAlgorithms<P, S, M> {

    /**
     * The maximum Score of the maximizing player.
     *
     * @param depth            The depth of search tree.
     * @param currentPlayer    The player who has to do the next move.
     * @param otherPlayer      The other player who has done the last move.
     * @param evaluatingPlayer The player for whom the game situation is to be evaluated.
     * @param generator        The generator that creates next moves.
     * @param state            The current state.
     * @param evaluator        The abstract evaluator.
     * @return The maximizing score that represents the best move of maximizing player.
     * @throws GameException
     */
    public Double maxScore(final int depth, final P currentPlayer, final P otherPlayer, final P evaluatingPlayer,
            final NextMoveGenerator<P, S, M> generator, final S state, final Evaluation<P, S> evaluator)
            throws GameException {
        final Collection<M> allNextMoves = generator.getAllNextMoves(state, currentPlayer);
        if (depth <= 0 || allNextMoves.isEmpty()) {
            return evaluator.evaluate(evaluatingPlayer, state);
        }
        Double bestScore = Double.NEGATIVE_INFINITY;

        final Iterator<M> it = allNextMoves.iterator();
        while (it.hasNext()) {
            final M currentMove = it.next();
            final S stateCopy = state.deepCopy();
            currentMove.applyTo(stateCopy, currentPlayer);
            stateCopy.nextTurn();
            final Double val = this
                    .minScore(depth - 1, otherPlayer, currentPlayer, evaluatingPlayer, generator, stateCopy, evaluator);
            if (val > bestScore) {
                bestScore = val;
            }
        }
        return bestScore;
    }

    /**
     * The minimum Score of the minimizing player.
     *
     * @param depth            The depth of search tree.
     * @param currentPlayer    The player who has to do the next move.
     * @param otherPlayer      The other player who has done the last move.
     * @param evaluatingPlayer The player for whom the game situation is to be evaluated.
     * @param generator        The generator which creates next moves.
     * @param state            The current state.
     * @param evaluator        The abstract evaluator.
     * @return The minimizing score that represents the best move of minimizing player.
     * @throws GameException
     */
    public Double minScore(final int depth, final P currentPlayer, final P otherPlayer, final P evaluatingPlayer,
            final NextMoveGenerator<P, S, M> generator, final S state, final Evaluation<P, S> evaluator)
            throws GameException {
        final Collection<M> allNextMoves = generator.getAllNextMoves(state, currentPlayer);
        if (depth <= 0 || allNextMoves.isEmpty()) {
            return evaluator.evaluate(evaluatingPlayer, state);
        }

        Double bestScore = Double.POSITIVE_INFINITY;
        final Iterator<M> it = allNextMoves.iterator();
        while (it.hasNext()) {
            final M currentMove = it.next();
            final S stateCopy = state.deepCopy();
            currentMove.applyTo(stateCopy, currentPlayer);
            stateCopy.nextTurn();
            final Double val = this
                    .maxScore(depth - 1, otherPlayer, currentPlayer, evaluatingPlayer, generator, stateCopy, evaluator);
            if (val < bestScore) {
                bestScore = val;
            }
        }
        return bestScore;
    }

    @Override
    public Optional<M> getBestMove(final int depth, final P currentPlayer, final NextMoveGenerator<P, S, M> generator,
            final S state, final Evaluation<P, S> evaluator) throws GameException {
        final Collection<MoveEvaluationPair<P, S, M>> moveBewertungen = new ArrayList<>();
        final P otherPlayer = state.getPlayers().values().stream().filter((final P p) -> !p.equals(currentPlayer))
                .findAny().get();

        final Collection<M> allPossibleMoves = generator.getAllNextMoves(state, currentPlayer);

        for (final M move : allPossibleMoves) {
            final S stateCopy = state.deepCopy();
            move.applyTo(stateCopy, currentPlayer);
            stateCopy.nextTurn();
            moveBewertungen.add(
                    new MoveEvaluationPair<>(
                            move,
                            this.minScore(
                                    depth - 1,
                                    otherPlayer,
                                    currentPlayer,
                                    currentPlayer,
                                    generator,
                                    stateCopy,
                                    evaluator)));
        }
        if (moveBewertungen.isEmpty()) {
            System.out.println("empty");
        }
        return moveBewertungen.stream().max((p1, p2) -> p1.getEvaluation().compareTo(p2.getEvaluation()))
                .map(MoveEvaluationPair::getMove);
    }
}
