/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-demo.
 *
 * Ipspiel21-demo is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-demo is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-demo. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.demo.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel21.demo.domain.DemoPlayer;
import de.fhdw.gaming.ipspiel21.demo.domain.DemoState;
import de.fhdw.gaming.ipspiel21.evolution.EvolutionPlayerImpl;

/**
 * Implements {@link DemoPlayer}.
 */
final class DemoPlayerImpl extends EvolutionPlayerImpl implements DemoPlayer {

    /**
     * The associated game state.
     */
    private final DemoState gameState;
    /**
     * The name of this player.
     */
    private final String name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<Boolean, Map<Boolean, Double>> possibleOutcomes;

    /**
     * Creates a Demo player.
     *
     * @param gameState        The associated game state.
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    DemoPlayerImpl(final DemoState gameState, final String name,
            final Map<Boolean, Map<Boolean, Double>> possibleOutcomes) {
        this.gameState = gameState;
        this.name = Objects.requireNonNull(name, "name");
        this.possibleOutcomes = Collections
                .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
    }

    @Override
    public String toString() {
        return String
                .format("DemoPlayer[name=%s, state=%s, outcome=%s]", this.name, this.getState(), this.getOutcome());
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof DemoPlayerImpl) {
            final DemoPlayerImpl other = (DemoPlayerImpl) obj;
            return this.name.equals(other.name) && this.getState().equals(other.getState())
                    && this.getOutcome().equals(other.getOutcome());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return this.name.hashCode();
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public Map<Boolean, Map<Boolean, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public PlayerState getState() {
        return this.gameState.getPlayerState(this.name);
    }

    @Override
    public Optional<Double> getOutcome() {
        return this.gameState.getPlayerOutcome(this.name);
    }

    @Override
    public DemoPlayer deepCopy(final DemoState newGameState) {
        return new DemoPlayerImpl(newGameState, this.name, this.possibleOutcomes);
    }
}
