/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-demo.
 *
 * Ipspiel21-demo is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-demo is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-demo. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.demo.domain.impl;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel21.demo.domain.DemoPlayer;
import de.fhdw.gaming.ipspiel21.demo.domain.DemoPlayerBuilder;
import de.fhdw.gaming.ipspiel21.demo.domain.DemoState;

/**
 * Implements {@link DemoState}.
 */
final class DemoStateImpl implements DemoState {

    /**
     * The first player.
     */
    private final DemoPlayer firstPlayer;
    /**
     * The second player.
     */
    private final DemoPlayer secondPlayer;
    /**
     * The states of the players.
     */
    private final Map<String, PlayerState> playerStates;
    /**
     * The outcomes of the players.
     */
    private final Map<String, Double> playerOutcomes;
    /**
     * The answers of the players.
     */
    private final Map<String, Boolean> playerAnswers;

    /**
     * Creates a Demo state.
     *
     * @param firstPlayerBuilder  A builder for the first player.
     * @param secondPlayerBuilder A builder for the second player.
     * @throws GameException if the state cannot be created according to the rules of the game.
     */
    DemoStateImpl(final DemoPlayerBuilder firstPlayerBuilder, final DemoPlayerBuilder secondPlayerBuilder)
            throws GameException {

        this.firstPlayer = Objects.requireNonNull(firstPlayerBuilder, "firstPlayerBuilder").build(this);
        this.secondPlayer = Objects.requireNonNull(secondPlayerBuilder, "secondPlayerBuilder").build(this);
        this.playerOutcomes = new LinkedHashMap<>();
        this.playerAnswers = new LinkedHashMap<>();

        this.playerStates = new LinkedHashMap<>();
        this.playerStates.put(this.firstPlayer.getName(), PlayerState.PLAYING);
        this.playerStates.put(this.secondPlayer.getName(), PlayerState.PLAYING);

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                    String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Creates a Demo state by copying an existing one.
     *
     * @param source The state to copy.
     */
    DemoStateImpl(final DemoStateImpl source) {
        this.firstPlayer = source.firstPlayer.deepCopy(this);
        this.secondPlayer = source.secondPlayer.deepCopy(this);
        this.playerOutcomes = new LinkedHashMap<>(source.playerOutcomes);
        this.playerAnswers = new LinkedHashMap<>(source.playerAnswers);
        this.playerStates = new LinkedHashMap<>();
        this.playerStates.put(this.firstPlayer.getName(), source.playerStates.get(this.firstPlayer.getName()));
        this.playerStates.put(this.secondPlayer.getName(), source.playerStates.get(this.secondPlayer.getName()));
    }

    /**
     * Returns the first player.
     */
    @Override
    public DemoPlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Returns the second player.
     */
    @Override
    public DemoPlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public String toString() {
        return String.format("DemoState[firstPlayer=%s, secondPlayer=%s]", this.firstPlayer, this.secondPlayer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof DemoStateImpl) {
            final DemoStateImpl other = (DemoStateImpl) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public DemoState deepCopy() {
        return new DemoStateImpl(this);
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    @Override
    public Map<String, DemoPlayer> getPlayers() {
        final Map<String, DemoPlayer> result = new LinkedHashMap<>();
        result.put(this.firstPlayer.getName(), this.firstPlayer);
        result.put(this.secondPlayer.getName(), this.secondPlayer);
        return result;
    }

    @Override
    public PlayerState getPlayerState(final String playerName) throws IllegalArgumentException {
        final PlayerState playerState = this.playerStates.get(playerName);
        if (playerState != null) {
            return playerState;
        } else {
            throw new IllegalArgumentException(this.buildUnknownPlayerMessage(playerName));
        }
    }

    @Override
    public void setPlayerState(final String playerName, final PlayerState newState) throws IllegalArgumentException {
        if (this.playerStates.containsKey(playerName)) {
            this.playerStates.put(playerName, newState);
            if (newState.equals(PlayerState.PLAYING)) {
                this.playerOutcomes.remove(playerName);
            }
        } else {
            throw new IllegalArgumentException(this.buildUnknownPlayerMessage(playerName));
        }
    }

    @Override
    public Optional<Double> getPlayerOutcome(final String playerName) throws IllegalArgumentException {
        if (this.playerStates.containsKey(playerName)) {
            final Double outcome = this.playerOutcomes.get(playerName);
            return outcome != null ? Optional.of(outcome) : DemoState.super.getPlayerOutcome(playerName);
        } else {
            throw new IllegalArgumentException(this.buildUnknownPlayerMessage(playerName));
        }
    }

    @Override
    public void setPlayerOutcome(final String playerName, final double newOutcome) throws IllegalArgumentException {
        if (this.getPlayerState(playerName).equals(PlayerState.PLAYING)) {
            throw new IllegalArgumentException(String.format("Cannot set outcome for player %s.", playerName));
        } else {
            this.playerOutcomes.put(playerName, newOutcome);
        }
    }

    @Override
    public Set<DemoPlayer> computeNextPlayers() {
        final Set<DemoPlayer> playersWithoutMove = new LinkedHashSet<>();
        if (!this.playerAnswers.containsKey(this.firstPlayer.getName())) {
            playersWithoutMove.add(this.firstPlayer);
        }
        if (!this.playerAnswers.containsKey(this.secondPlayer.getName())) {
            playersWithoutMove.add(this.secondPlayer);
        }
        return playersWithoutMove;
    }

    @Override
    public void nextTurn() {
        final Set<DemoPlayer> playersWithoutMove = this.computeNextPlayers();
        if (playersWithoutMove.isEmpty()) {
            final Boolean answerOfFirstPlayer = this.playerAnswers.get(this.firstPlayer.getName());
            final Boolean answerOfSecondPlayer = this.playerAnswers.get(this.secondPlayer.getName());

            final Double outcomeOfFirstPlayer = this.firstPlayer.getPossibleOutcomes().get(answerOfFirstPlayer)
                    .get(answerOfSecondPlayer);
            this.setPlayerState(this.firstPlayer.getName(), outcomeToState(outcomeOfFirstPlayer));
            this.setPlayerOutcome(this.firstPlayer.getName(), outcomeOfFirstPlayer);

            final Double outcomeOfSecondPlayer = this.secondPlayer.getPossibleOutcomes().get(answerOfFirstPlayer)
                    .get(answerOfSecondPlayer);
            this.setPlayerState(this.secondPlayer.getName(), outcomeToState(outcomeOfSecondPlayer));
            this.setPlayerOutcome(this.secondPlayer.getName(), outcomeOfSecondPlayer);
        }
    }

    /**
     * Computes a player state from an outcome.
     *
     * @param outcome The player's outcome.
     */
    private static PlayerState outcomeToState(final Double outcome) {
        return outcome > 0.0 ? PlayerState.WON : outcome < 0.0 ? PlayerState.LOST : PlayerState.DRAW;
    }

    @Override
    public void setAnswer(final DemoPlayer player, final boolean answer) {
        if (this.playerStates.containsKey(player.getName())) {
            final Boolean oldAnswer = this.playerAnswers.put(player.getName(), answer);
            if (oldAnswer != null) {
                throw new IllegalArgumentException(
                        String.format("Player %s tried to change her answer.", player.getName()));
            }
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", player.getName()));
        }
    }

    /**
     * Returns a message about a player being unknown to the game.
     *
     * @param playerName The name of the unknown player.
     */
    private String buildUnknownPlayerMessage(final String playerName) {
        return String.format("Unknown player %s.", playerName);
    }
}
