package de.fhdw.gaming.ipspiel21.dilemmaCustomized.impl;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.ipspiel21.customCore.core.domain.CustomGameBuilderFactory;
import de.fhdw.gaming.ipspiel21.customGui.gui.CustomInputProvider;
import de.fhdw.gaming.ipspiel21.dilemmaBasic.DilemmaBasicGameBuilderFactoryImpl;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaGameBuilder;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaStrategy;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.impl.DilemmaGameBuilderFactoryImpl;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.DilemmaMove;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.factory.DilemmaMoveFactory;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.DilemmaDefaultMoveFactory;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.strategy.DilemmaCustomStrategy;

/**
 * 
 * extends {@link DilemmaGameBuilderFactoryImpl} by creating an extended input
 * provider. implements {@link CustomizedGameBuilderFactory} by .
 * 
 */
public class DilemmaCustomizedGameBuilderFactoryImpl implements GameBuilderFactory, CustomGameBuilderFactory {

    /**
     * All available dilemma moves.
     */
    private final Set<DilemmaMove> moves;
    /**
     * .
     */
    private final DilemmaGameBuilderFactoryImpl originalGameBuilder;
    

    /**
     * Creates a dilemma game factory.
     */
    public DilemmaCustomizedGameBuilderFactoryImpl() {
        final DilemmaMoveFactory moveFactory = new DilemmaDefaultMoveFactory();
        this.moves = new LinkedHashSet<>();
        this.moves.add(moveFactory.createNoMove());
        this.moves.add(moveFactory.createYesMove());
        this.originalGameBuilder = new DilemmaBasicGameBuilderFactoryImpl();
    }

    @Override
    public List<? extends Move<?, ?>> getMoves() {
        return new ArrayList<>(this.moves);
    }

    @Override
    public String getName() {
        return this.originalGameBuilder.getName() + "Customized";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return originalGameBuilder.getMinimumNumberOfPlayers();
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return originalGameBuilder.getMaximumNumberOfPlayers();
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return originalGameBuilder.getStrategies();
    }

    @Override
    public GameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final DilemmaGameBuilder gameBuilder = originalGameBuilder.createGameBuilder(inputProvider);

            final DilemmaStrategy firstPlayerStrategy = originalGameBuilder.getActualStrategyPlayer1();

            if (firstPlayerStrategy instanceof DilemmaCustomStrategy) {
                final Map<String, Object> strategyProperties = this.requestPlayerStrategyProperties(inputProvider,
                        "Strategy-Properties");

                ((DilemmaCustomStrategy) firstPlayerStrategy).setAmountOfGames(
                        (Integer) strategyProperties.get(CustomGameBuilderFactory.PARAM_AMOUNT_GAMES));
                ((DilemmaCustomStrategy) firstPlayerStrategy).setInitialMove(
                        (DilemmaMove) strategyProperties.get(CustomGameBuilderFactory.PARAM_PLAYER_MOVE));

                final InputProvider numberBasedInputProvider = inputProvider.getNext(strategyProperties);
                final Map<String, Object> configuredStrategyDataNumberBased = this.requestPlayerStrategyNumberBased(
                        numberBasedInputProvider, "Number based set up - Strategy-Properties");
                ((DilemmaCustomStrategy) firstPlayerStrategy).setProvidedMoveData(configuredStrategyDataNumberBased);
                originalGameBuilder.setActualStrategyPlayer1(firstPlayerStrategy);

            }
            final DilemmaStrategy secondPlayerStrategy = originalGameBuilder.getActualStrategyPlayer2();

            if (secondPlayerStrategy instanceof DilemmaCustomStrategy) {
                final Map<String, Object> strategyProperties = this.requestPlayerStrategyProperties(inputProvider,
                        "Strategy-Properties");

                ((DilemmaCustomStrategy) secondPlayerStrategy).setAmountOfGames(
                        (Integer) strategyProperties.get(CustomGameBuilderFactory.PARAM_AMOUNT_GAMES));
                ((DilemmaCustomStrategy) secondPlayerStrategy).setInitialMove(
                        (DilemmaMove) strategyProperties.get(CustomGameBuilderFactory.PARAM_PLAYER_MOVE));

                final InputProvider numberBasedInputProvider = inputProvider.getNext(strategyProperties);
                final Map<String, Object> configuredStrategyDataNumberBased = this.requestPlayerStrategyNumberBased(
                        numberBasedInputProvider, "Number based set up - Strategy-Properties");
                ((DilemmaCustomStrategy) secondPlayerStrategy).setProvidedMoveData(configuredStrategyDataNumberBased);
                originalGameBuilder.setActualStrategyPlayer2(secondPlayerStrategy);
            }
            return new DilemmaCustomizedGameBuilderWrapper(gameBuilder);
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating dilemma game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * 
     * Returns data for the amount of considered games and the initial move choice
     * for the custom Strategy.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @return
     * @throws GameException
     * @throws InputProviderException
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerStrategyProperties(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needInteger(CustomGameBuilderFactory.PARAM_AMOUNT_GAMES, "Amount of considered last games",
                        Optional.empty(), new MinValueValidator<Integer>(2), new MaxValueValidator<Integer>(10))
                .needObject(CustomGameBuilderFactory.PARAM_PLAYER_MOVE, "Initial move", Optional.empty(), this.moves);

        return inputProvider.requestData(title);

    }

    /**
     * 
     * Returns data for the number based Configuration of the custom Strategy.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @return
     * @throws GameException
     * @throws InputProviderException
     */
    private  Map<String, Object> requestPlayerStrategyNumberBased(final InputProvider inputProvider,
            final String title) throws InputProviderException {

        for (final DilemmaMove dilemmaMove : moves) {
            inputProvider
                    .needObject(dilemmaMove.toString() + " Numberbased100",
                            "100% " + dilemmaMove.toString() + " and my choice is: ", Optional.empty(), moves)
                    .needObject(dilemmaMove.toString() + " Numberbased50",
                            ">50% " + dilemmaMove.toString() + " and my choice is: ", Optional.empty(), moves);
        }
        return inputProvider.requestData(title);
    }

    @Override
    public InputProvider extendInputProvider(final InputProvider inputProvider) throws InputProviderException {
        return new CustomInputProvider(inputProvider);
    }

}
