/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-freizeit.
 *
 * Ipspiel21-freizeit is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-freizeit is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied  * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-freizeit. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.freizeit.domain.impl;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultObserverFactoryProvider;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitGame;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitGameBuilder;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitPlayerBuilder;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitStrategy;
import de.fhdw.gaming.ipspiel21.freizeit.moves.impl.AbstractFreizeitMove;

/**
 * Implements {@link FreizeitGameBuilder}.
 */
final class FreizeitGameBuilderImpl implements FreizeitGameBuilder {

    /**
     * The {@link ObserverFactoryProvider}.
     */
    private ObserverFactoryProvider observerFactoryProvider;
    /**
     * The builder for the player using black tokens.
     */
    private Optional<FreizeitPlayerBuilder> firstPlayer;
    /**
     * The strategy of the player using black tokens.
     */
    private Optional<FreizeitStrategy> firstPlayerStrategy;
    /**
     * The builder for the player using white tokens.
     */
    private Optional<FreizeitPlayerBuilder> secondPlayer;
    /**
     * The strategy of the player using white tokens.
     */
    private Optional<FreizeitStrategy> secondPlayerStrategy;
    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Creates a Freizeit game builder.
     */
    FreizeitGameBuilderImpl() {
        this.observerFactoryProvider = new DefaultObserverFactoryProvider();
        this.firstPlayer = Optional.empty();
        this.firstPlayerStrategy = Optional.empty();
        this.secondPlayer = Optional.empty();
        this.secondPlayerStrategy = Optional.empty();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public FreizeitPlayerBuilder createPlayerBuilder() {
        return new FreizeitPlayerBuilderImpl();
    }

    @Override
    public FreizeitGameBuilder addPlayerBuilder(final FreizeitPlayerBuilder playerBuilder,
            final FreizeitStrategy strategy) throws GameException {

        Objects.requireNonNull(playerBuilder);
        if (this.firstPlayer.isEmpty()) {
            this.firstPlayer = Optional.of(playerBuilder);
            this.firstPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "firstPlayerStrategy"));
        } else if (this.secondPlayer.isEmpty()) {
            this.secondPlayer = Optional.of(playerBuilder);
            this.secondPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "secondPlayerStrategy"));
        } else {
            throw new GameException(String.format("More than two players are now allowed."));
        }
        return this;
    }

    @Override
    public FreizeitGameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public FreizeitGameBuilder changeObserverFactoryProvider(final ObserverFactoryProvider newObserverFactoryProvider) {
        this.observerFactoryProvider = newObserverFactoryProvider;
        return this;
    }

    @Override
    public FreizeitGame build(final int id) throws GameException, InterruptedException {
        if (!this.firstPlayer.isPresent() || !this.secondPlayer.isPresent()) {
            throw new GameException("A Freizeit game needs two players.");
        }

        final FreizeitStateImpl initialState = new FreizeitStateImpl(this.firstPlayer.get(), this.secondPlayer.get());

        final Map<String, FreizeitStrategy> strategies = new LinkedHashMap<>();
        strategies.put(initialState.getFirstPlayer().getName(), this.firstPlayerStrategy.orElseThrow());
        strategies.put(initialState.getSecondPlayer().getName(), this.secondPlayerStrategy.orElseThrow());
        return new FreizeitGameImpl(
                id,
                initialState,
                strategies,
                this.maxComputationTimePerMove,
                AbstractFreizeitMove.class::isInstance,
                this.observerFactoryProvider);
    }
}
