/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-freizeit.
 *
 * Ipspiel21-freizeit is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-freizeit is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied  * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-freizeit. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.freizeit.domain.impl;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitPlayer;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitPlayerBuilder;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitState;
import de.fhdw.gaming.ipspiel21.freizeit.moves.impl.AbstractFreizeitMove;

/**
 * Implements {@link FreizeitPlayerBuilder}.
 */
final class FreizeitPlayerBuilderImpl implements FreizeitPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<AbstractFreizeitMove, Map<AbstractFreizeitMove, Double>>> possibleOutcomes;

    /**
     * Creates an {@link FreizeitPlayerBuilderImpl}.
     */
    FreizeitPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public FreizeitPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public FreizeitPlayerBuilder changePossibleOutcomes(
            final Map<AbstractFreizeitMove, Map<AbstractFreizeitMove, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public FreizeitPlayer build(final FreizeitState state) throws GameException {
        return new FreizeitPlayerImpl(
                state,
                this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are correctly defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<AbstractFreizeitMove, Map<AbstractFreizeitMove, Double>> checkPossibleOutcomes(
            final Map<AbstractFreizeitMove, Map<AbstractFreizeitMove, Double>> outcomes) {

        if (!isNullOrEmptyOuterMap(outcomes) && !isNullOrEmptyInnerMap(outcomes.values())) {
            return outcomes;
        } else {
            throw new IllegalArgumentException();
        }
    }

    /**
     * Check if the outer map of possibleOutcomes is empty or has a null value.
     *
     * @param outcomes possible Outcomes map.
     * @return false if the condition are not fulfilled, else return an IllegalArgumentException.
     */
    private static boolean isNullOrEmptyOuterMap(
            final Map<AbstractFreizeitMove, Map<AbstractFreizeitMove, Double>> outcomes) {
        if (outcomes.isEmpty()) {
            throw new IllegalArgumentException("The outcomes are not defined");
        } else {
            final Iterator<Map<AbstractFreizeitMove, Double>> it = outcomes.values().iterator();
            while (it.hasNext()) {
                if (it.next() == null) {
                    throw new IllegalArgumentException("The value is not defined: " + it.next());
                }
            }
            return false;
        }
    }

    /**
     * Check if the inner map of possibleOutcomes is empty or has a null value.
     *
     * @param outcomes inner map of possibleOutocomes.
     * @return false if the condition are not fulfilled, else return an IllegalArgumentException.
     */
    private static boolean isNullOrEmptyInnerMap(final Collection<Map<AbstractFreizeitMove, Double>> outcomes) {
        if (outcomes.isEmpty()) {
            throw new IllegalArgumentException("The outcomes are not defined");
        } else {
            final Iterator<Map<AbstractFreizeitMove, Double>> it = outcomes.iterator();
            while (it.hasNext()) {
                if (it.next().values() == null) {
                    throw new IllegalArgumentException("The value is not defined: " + it.next());
                }
            }
            return false;
        }
    }
}
