/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-freizeit.
 *
 * Ipspiel21-freizeit is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-freizeit is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied  * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-freizeit. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.freizeit.domain.impl;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitPlayer;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitPlayerBuilder;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitState;
import de.fhdw.gaming.ipspiel21.freizeit.moves.impl.AbstractFreizeitMove;

/**
 * Implements {@link FreizeitState}.
 */
@SuppressWarnings("PMD.GodClass")
final class FreizeitStateImpl implements FreizeitState {
    /**
     * The first player.
     */
    private final FreizeitPlayer firstPlayer;
    /**
     * The second player.
     */
    private final FreizeitPlayer secondPlayer;
    /**
     * The moves.
     */
    private final Map<FreizeitPlayer, AbstractFreizeitMove> playerMove;
    /**
     * The states.
     */
    private final Map<String, PlayerState> playerStateMap;

    /**
     * Map for the outcome of the Players.
     */
    private final Map<String, Double> outcomePlayerMap;

    /**
     * Creates a Freizeit state.
     *
     * @param firstPlayerBuilder  A builder for the first player.
     * @param secondPlayerBuilder A builder for the second player.
     * @throws GameException if the state cannot be created according to the rules of the game.
     */
    FreizeitStateImpl(final FreizeitPlayerBuilder firstPlayerBuilder, final FreizeitPlayerBuilder secondPlayerBuilder)
            throws GameException {

        this.firstPlayer = Objects.requireNonNull(firstPlayerBuilder, "firstPlayerBuilder").build(this);
        this.secondPlayer = Objects.requireNonNull(secondPlayerBuilder, "secondPlayerBuilder").build(this);

        this.outcomePlayerMap = new LinkedHashMap<>();

        this.playerStateMap = new LinkedHashMap<>();
        this.playerStateMap.put(this.firstPlayer.getName(), PlayerState.PLAYING);
        this.playerStateMap.put(this.secondPlayer.getName(), PlayerState.PLAYING);

        this.playerMove = new LinkedHashMap<>();

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                    String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Creates a Freizeit state by copying an existing one.
     *
     * @param source The state to copy.
     */
    FreizeitStateImpl(final FreizeitStateImpl source) {
        this.firstPlayer = source.firstPlayer.deepCopy(this);
        this.secondPlayer = source.secondPlayer.deepCopy(this);
        this.outcomePlayerMap = new LinkedHashMap<>(source.outcomePlayerMap);
        this.playerStateMap = new LinkedHashMap<>();
        this.playerStateMap.put(this.firstPlayer.getName(), source.playerStateMap.get(this.firstPlayer.getName()));
        this.playerStateMap.put(this.secondPlayer.getName(), source.playerStateMap.get(this.secondPlayer.getName()));
        this.playerMove = new LinkedHashMap<>();
    }

    /**
     * Returns the first player.
     */
    @Override
    public FreizeitPlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Returns the second player.
     */
    @Override
    public FreizeitPlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public String toString() {
        return String.format("FreizeitState[firstPlayer=%s, secondPlayer=%s]", this.firstPlayer, this.secondPlayer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof FreizeitStateImpl) {
            final FreizeitStateImpl other = (FreizeitStateImpl) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public FreizeitState deepCopy() {
        return this;
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    @Override
    public Map<String, FreizeitPlayer> getPlayers() {
        final Map<String, FreizeitPlayer> result = new LinkedHashMap<>();
        result.put(this.firstPlayer.getName(), this.firstPlayer);
        result.put(this.secondPlayer.getName(), this.secondPlayer);
        return result;
    }

    @Override
    public PlayerState getPlayerState(final String playerName) throws IllegalArgumentException {
        final PlayerState playerState = this.playerStateMap.get(playerName);
        if (playerState != null) {
            return playerState;
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
    }

    @Override
    public void setPlayerState(final String playerName, final PlayerState newState) throws IllegalArgumentException {
        if (this.playerStateMap.containsKey(playerName)) {
            this.playerStateMap.put(playerName, newState);
            if (newState.equals(PlayerState.PLAYING)) {
                this.outcomePlayerMap.remove(playerName);
            }
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
    }

    @Override
    public Optional<Double> getPlayerOutcome(final String playerName) throws IllegalArgumentException {
        if (this.playerStateMap.containsKey(playerName)) {
            final Double outcome = this.outcomePlayerMap.get(playerName);
            return outcome != null ? Optional.of(outcome) : FreizeitState.super.getPlayerOutcome(playerName);
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
    }

    @Override
    public void setPlayerOutcome(final String playerName, final double newOutcome) throws IllegalArgumentException {
        if (this.playerStateMap.containsKey(playerName)) {
            this.outcomePlayerMap.put(playerName, newOutcome);
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s was found.", playerName));
        }
    }

    @Override
    public AbstractFreizeitMove getPlayerMove(final FreizeitPlayer player) {
        return this.playerMove.get(player);
    }

    @Override
    public Set<FreizeitPlayer> computeNextPlayers() {
        final Set<FreizeitPlayer> playersWithoutAnswer = new LinkedHashSet<>();
        if (!this.playerMove.containsKey(this.getFirstPlayer())) {
            playersWithoutAnswer.add(this.firstPlayer);
        }
        if (!this.playerMove.containsKey(this.getSecondPlayer())) {
            playersWithoutAnswer.add(this.secondPlayer);
        }
        return playersWithoutAnswer;
    }

    @Override
    public void nextTurn() {
        final Set<FreizeitPlayer> playersWithoutAnswer = this.computeNextPlayers();
        if (playersWithoutAnswer.isEmpty()) {

            this.setNewOutcomes(this.firstPlayer, this.secondPlayer);
            this.setNewOutcomes(this.secondPlayer, this.firstPlayer);

            this.setNewPlayerState(this.firstPlayer);
            this.setNewPlayerState(this.secondPlayer);
        }
    }

    /**
     * Set the new outcome depending on the moves of a player.
     *
     * @param firstTurn  The first player.
     * @param secondTurn The second player.
     */
    private void setNewOutcomes(final FreizeitPlayer firstTurn, final FreizeitPlayer secondTurn) {
        final AbstractFreizeitMove firstPlayerMove = this.getPlayerMove(firstTurn);
        final AbstractFreizeitMove secondPlayerMove = this.getPlayerMove(secondTurn);
        final Double outcome = firstTurn.getPossibleOutcomes().get(firstPlayerMove).get(secondPlayerMove);

        this.setPlayerOutcome(firstTurn.getName(), outcome);

    }

    /**
     * Set the states of the player depending on the result of the game.
     *
     * @param player The player.
     */
    private void setNewPlayerState(final FreizeitPlayer player) {

        if (player == null) {
            throw new IllegalArgumentException(String.format("The player was not found", player));
        }

        final Optional<FreizeitPlayer> otherPlayer = this.getPlayers().values().stream()
                .filter(player1 -> !player.equals(player1)).findAny();

        final Optional<Double> outcome1Option = player.getOutcome();
        final Optional<Optional<Double>> outcome2Option = otherPlayer.map(FreizeitPlayer::getOutcome);

        if (outcome1Option.isEmpty() || outcome2Option.isEmpty() || outcome2Option.get().isEmpty()) {
            throw new IllegalArgumentException("At least one of the outcomes of the players was not found");
        }

        final Double outcome1 = outcome1Option.get();
        final Double outcome2 = outcome2Option.get().get();

        if (outcome1 > outcome2) {
            this.setPlayerState(player.getName(), PlayerState.WON);
        } else if (outcome1 < outcome2) {
            this.setPlayerState(player.getName(), PlayerState.LOST);
        } else {
            this.setPlayerState(player.getName(), PlayerState.DRAW);
        }
    }

    @Override
    public void setAnswer(final FreizeitPlayer player, final AbstractFreizeitMove answer) {
        try {
            if (!this.playerMove.containsKey(player)) {
                this.playerMove.put(player, answer);
            } else {
                throw new IllegalArgumentException(String.format("Player %s have already given an answer", player));
            }
        } catch (final Exception e) {
            throw new IllegalArgumentException(e);
        }
    }

}
