/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-Kopfzahlkate.
 *
 * Ipspiel21-Kopfzahlkante is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-Kopfzahlkante is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-Kopfzahlkante. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.impl;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.DefaultObserverFactoryProvider;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkanteGame;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkanteGameBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkantePlayerBuilder;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.domain.KopfzahlkanteStrategy;
import de.fhdw.gaming.ipspiel21.kopfzahlkante.moves.impl.AbstractKopfzahlkanteMove;

/**
 * Implements {@link KopfzahlkanteGameBuilder}.
 */
final class KopfzahlkanteGameBuilderImpl implements KopfzahlkanteGameBuilder {

    /**
     * The {@link ObserverFactoryProvider}.
     */
    private ObserverFactoryProvider observerFactoryProvider;
    /**
     * The builder for the player using black tokens.
     */
    private Optional<KopfzahlkantePlayerBuilder> firstPlayer;
    /**
     * The strategy of the player using black tokens.
     */
    private Optional<KopfzahlkanteStrategy> firstPlayerStrategy;
    /**
     * The builder for the player using white tokens.
     */
    private Optional<KopfzahlkantePlayerBuilder> secondPlayer;
    /**
     * The strategy of the player using white tokens.
     */
    private Optional<KopfzahlkanteStrategy> secondPlayerStrategy;
    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Creates a Kopfzahlkante game builder.
     */
    KopfzahlkanteGameBuilderImpl() {
        this.observerFactoryProvider = new DefaultObserverFactoryProvider();
        this.firstPlayer = Optional.empty();
        this.firstPlayerStrategy = Optional.empty();
        this.secondPlayer = Optional.empty();
        this.secondPlayerStrategy = Optional.empty();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public KopfzahlkantePlayerBuilder createPlayerBuilder() {
        return new KopfzahlkantePlayerBuilderImpl();
    }

    @Override
    public KopfzahlkanteGameBuilder addPlayerBuilder(final KopfzahlkantePlayerBuilder playerBuilder,
            final KopfzahlkanteStrategy strategy) throws GameException {

        Objects.requireNonNull(playerBuilder);
        if (this.firstPlayer.isEmpty()) {
            this.firstPlayer = Optional.of(playerBuilder);
            this.firstPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "firstPlayerStrategy"));
        } else if (this.secondPlayer.isEmpty()) {
            this.secondPlayer = Optional.of(playerBuilder);
            this.secondPlayerStrategy = Optional.of(Objects.requireNonNull(strategy, "secondPlayerStrategy"));
        } else {
            throw new GameException(String.format("More than two players are now allowed."));
        }
        return this;
    }

    @Override
    public KopfzahlkanteGameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public KopfzahlkanteGameBuilder changeObserverFactoryProvider(
            final ObserverFactoryProvider newObserverFactoryProvider) {
        this.observerFactoryProvider = newObserverFactoryProvider;
        return this;
    }

    @Override
    public KopfzahlkanteGame build(final int id) throws GameException, InterruptedException {
        if (!this.firstPlayer.isPresent() || !this.secondPlayer.isPresent()) {
            throw new GameException("A Kopfzahlkante game needs two players.");
        }

        final KopfzahlkanteStateImpl initialState = new KopfzahlkanteStateImpl(this.firstPlayer.get(),
                this.secondPlayer.get());

        final Map<String, KopfzahlkanteStrategy> strategies = new LinkedHashMap<>();
        strategies.put(initialState.getFirstPlayer().getName(), this.firstPlayerStrategy.orElseThrow());
        strategies.put(initialState.getSecondPlayer().getName(), this.secondPlayerStrategy.orElseThrow());
        return new KopfzahlkanteGameImpl(id, initialState, strategies, this.maxComputationTimePerMove,
                AbstractKopfzahlkanteMove.class::isInstance, this.observerFactoryProvider);
    }
}
