/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-tictactoe-gui.
 *
 * ipspiel21-tictactoe-gui is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel21-tictactoe-gui is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel21-tictactoe-gui. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.tictactoe.gui.impl;

import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToePlayer;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeState;
import de.fhdw.gaming.ipspiel21.tictactoe.core.domain.TicTacToeStrategy;
import de.fhdw.gaming.ipspiel21.tictactoe.core.moves.TicTacToeMove;
import de.fhdw.gaming.ipspiel21.tictactoe.core.moves.factory.TicTacToeMoveFactory;
import de.fhdw.gaming.ipspiel21.tictactoe.gui.TicTacToeBoardEventProvider;
import de.fhdw.gaming.ipspiel21.tictactoe.gui.event.TicTacToeBoardEventVisitor;
import de.fhdw.gaming.ipspiel21.tictactoe.gui.event.TicTacToeMakeMoveBoardEvent;

/**
 * Implements {@link TicTacToeStrategy} by asking the user for a move.
 */
public final class TicTacToeInteractiveStrategy implements TicTacToeStrategy {

    /**
     * The factory for creating TicTacToe moves.
     */
    private final TicTacToeMoveFactory moveFactory;

    /**
     * Creates an {@link TicTacToeInteractiveStrategy}.
     *
     * @param moveFactory The factory for creating TicTacToe moves.
     */
    TicTacToeInteractiveStrategy(final TicTacToeMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    /**
     * Asks the user to choose a move.
     */
    @Override
    public Optional<TicTacToeMove> computeNextMove(final int gameId, final TicTacToePlayer player,
            final TicTacToeState state) throws GameException {

        final Optional<TicTacToeBoardEventProvider> provider = TicTacToeGuiObserverImpl.getEventProvider(gameId);
        if (provider.isEmpty()) {
            return Optional.empty();
        }

        final AtomicReference<TicTacToeMove> move = new AtomicReference<>();
        provider.get().waitForEvent(player, state).accept(new TicTacToeBoardEventVisitor() {

            @Override
            public void handleMakeMove(final TicTacToeMakeMoveBoardEvent event) {
                move.setPlain(
                        TicTacToeInteractiveStrategy.this.moveFactory
                                .createPlaceMarkMove(player.isUsingCrosses(), event.getFieldPosition()));
            }
        });

        return Optional.ofNullable(move.getPlain());
    }

    @Override
    public String toString() {
        return TicTacToeInteractiveStrategy.class.getSimpleName();
    }

    @Override
    public boolean isInteractive() {
        return true;
    }

    @Override
    public void abortRequested(final int gameId) {
        final Optional<TicTacToeBoardEventProvider> provider = TicTacToeGuiObserverImpl.getEventProvider(gameId);
        if (!provider.isEmpty()) {
            provider.get().cancelWaiting();
        }
    }
}
