package de.fhdw.gaming.ipspiel21.viergewinnt.strategies.group2.strategy;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import de.fhdw.gaming.ipspiel21.searchtrees.domain.Evaluation;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntDirection;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntField;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntFieldState;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntPlayer;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntState;

/**
 * .
 *
 * @author Robby Rabbitman
 *
 */
public class VierGewinntEvaluation implements Evaluation<VierGewinntPlayer, VierGewinntState> {

    /**
     * Eval Map for Points given in Vier-Gewinnt Game.
     *
     */
    private static final Map<Integer, Double> EVALUATION_MAP = Map.of(1, 1.0, 2, 50.0, 3, 300.0, 4, 10_000_000.0);

    @Override
    public Double evaluate(final VierGewinntPlayer player, final VierGewinntState state) {
        return this.evaluatePlayer(player, state);
    }

    /**
     * Evaluate the Points for a given Player for a given State(Game Field). *
     * 
     * @param player
     * @param state
     * @return Points for specific Player
     */
    private Double evaluatePlayer(final VierGewinntPlayer player, final VierGewinntState state) {
        return state.getBoard().getFieldsBeing(player.getPlayerMark()).values().stream()
                .map(field -> this.getRows(field, 4))
                .map(rows -> rows.stream()
                        .mapToDouble(row -> this.evaluateRow(row, player.getPlayerMark(),
                                this.getOtherPlayer(player, state).getPlayerMark()))
                        .sum())
                .reduce(0.0, Double::sum); // sum up all values
    }

    /**
     * Get the Opponent Player. Need current PLayer and GameState.
     *
     * @param player
     * @param state
     */
    private VierGewinntPlayer getOtherPlayer(final VierGewinntPlayer player, final VierGewinntState state) {
        if (player.isUsingYellow()) {
            return state.getRedPlayer();
        } else {
            return state.getYellowPlayer();
        }
    }

    /**
     * Evaluate the Points for a PlayerState for a list of rows.
     *
     * @param row
     * @param playerState
     * @param otherPlayerState
     * @return
     */
    private Double evaluateRow(final List<VierGewinntFieldState> row, final VierGewinntFieldState playerState,
            final VierGewinntFieldState otherPlayerState) {
        if (row.contains(otherPlayerState) || row.size() < 4) {
            return 0.0;
        } else {
            return VierGewinntEvaluation.EVALUATION_MAP.get(this.numberOfFieldState(row, playerState));
        }
    }

    /**
     * Return the Number of marked Fields in Gamefield.
     *
     * @param row
     * @param state
     * @return
     */
    private Integer numberOfFieldState(final List<VierGewinntFieldState> row, final VierGewinntFieldState state) {
        return (int) row.stream().filter(x -> x.equals(state)).count();
    }

    /**
     * Return all adjacent fields of a given field. They will collected in a list.
     *
     * @param field
     * @param radius
     * @return
     */
    private List<List<VierGewinntFieldState>> getRows(final VierGewinntField field, final Integer radius) {
        final List<List<VierGewinntFieldState>> rows = new ArrayList<>();
        rows.add(this.getRowsHelper(field, radius, VierGewinntDirection.NORTHWEST));
        rows.add(this.getRowsHelper(field, radius, VierGewinntDirection.SOUTHEAST));
        rows.add(this.getRowsHelper(field, radius, VierGewinntDirection.NORTHEAST));
        rows.add(this.getRowsHelper(field, radius, VierGewinntDirection.SOUTHWEST));
        rows.add(this.getRowsHelper(field, radius, VierGewinntDirection.NORTH));
        rows.add(this.getRowsHelper(field, radius, VierGewinntDirection.SOUTH));
        rows.add(this.getRowsHelper(field, radius, VierGewinntDirection.WEST));
        rows.add(this.getRowsHelper(field, radius, VierGewinntDirection.EAST));
        return rows;
    }

    /**
     * Helper to build rows.
     *
     * @param field
     * @param radius
     * @param direction
     * @return
     */
    private List<VierGewinntFieldState> getRowsHelper(final VierGewinntField field, final Integer radius,
            final VierGewinntDirection direction) {
        final List<VierGewinntFieldState> out = new ArrayList<>();
        out.add(field.getState()); // always put first field into row
        VierGewinntField currentField = field;
        for (int i = 0; i < radius - 1; i++) {
            if (currentField.hasNeighbour(direction)) {
                currentField = currentField.getNeighbour(direction);
                out.add(currentField.getState());
            } else {
                return out;
            }
        }
        return out;
    }
}
