/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-freizeit.
 *
 * Ipspiel21-freizeit is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-freizeit is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied  * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-freizeit. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.freizeit.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel21.evolution.EvolutionPlayerImpl;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitPlayer;
import de.fhdw.gaming.ipspiel21.freizeit.domain.FreizeitState;
import de.fhdw.gaming.ipspiel21.freizeit.moves.impl.AbstractFreizeitMove;

/**
 * Implements {@link FreizeitPlayer}.
 */
final class FreizeitPlayerImpl extends EvolutionPlayerImpl implements FreizeitPlayer {
    /**
     * The associated game state.
     */
    private final FreizeitState gameState;
    /**
     * The name of this player.
     */
    private final String name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<AbstractFreizeitMove, Map<AbstractFreizeitMove, Double>> possibleOutcomes;

    /**
     * Creates a Freizeit player.
     *
     * @param gameState        The associated game state.
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    FreizeitPlayerImpl(final FreizeitState gameState, final String name,
            final Map<AbstractFreizeitMove, Map<AbstractFreizeitMove, Double>> possibleOutcomes) {
        this.gameState = gameState;
        this.name = Objects.requireNonNull(name, "name");
        this.possibleOutcomes = Collections
                .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
    }

    @Override
    public String toString() {
        return String
                .format("FreizeitPlayer[name=%s, state=%s, outcome=%s]", this.name, this.getState(), this.getOutcome());
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof FreizeitPlayerImpl) {
            final FreizeitPlayerImpl other = (FreizeitPlayerImpl) obj;
            return this.name.equals(other.name) && this.getState().equals(other.getState())
                    && this.getOutcome().equals(other.getOutcome());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return this.name.hashCode();
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public Map<AbstractFreizeitMove, Map<AbstractFreizeitMove, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public PlayerState getState() {
        return this.gameState.getPlayerState(this.name);
    }

    @Override
    public Optional<Double> getOutcome() {
        return this.gameState.getPlayerOutcome(this.name);
    }

    @Override
    public FreizeitPlayer deepCopy(final FreizeitState newGameState) {
        return new FreizeitPlayerImpl(newGameState, this.name, this.possibleOutcomes);
    }

}
