package de.fhdw.gaming.ipspiel21.searchtrees.domain.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.State;
import de.fhdw.gaming.ipspiel21.searchtrees.domain.Evaluation;
import de.fhdw.gaming.ipspiel21.searchtrees.domain.NextMoveGenerator;
import de.fhdw.gaming.ipspiel21.searchtrees.domain.SearchTreeAlgorithms;

/**
 *
 * Implements {@link SearchTreeAlgorithms} by implementation of the Negamax algorithm.
 *
 * @param <P>
 * @param <S>
 * @param <M>
 */
public class NegamaxImpl<P extends Player, S extends State<P, S>, M extends Move<P, S>>
        implements SearchTreeAlgorithms<P, S, M> {

    @Override
    public Optional<M> getBestMove(final int depth, final P currentPlayer, final NextMoveGenerator<P, S, M> generator,
            final S state, final Evaluation<P, S> evaluator) throws GameException {
        final P otherPlayer = state.getPlayers().values().stream().filter((final P p) -> !p.equals(currentPlayer))
                .findAny().get();
        final Collection<MoveEvaluationPair<P, S, M>> moveBewertungen = new ArrayList<>();
        final Collection<M> allPossibleMoves = currentPlayer == null ? Collections.emptyList()
                : generator.getAllNextMoves(state, currentPlayer);
        for (final M move : allPossibleMoves) {
            final S stateCopy = state.deepCopy();
            move.applyTo(stateCopy, currentPlayer);
            stateCopy.nextTurn();
            moveBewertungen.add(
                    new MoveEvaluationPair<>(
                            move,
                            -this.negamaxScore(
                                    depth - 1,
                                    otherPlayer,
                                    currentPlayer,
                                    generator,
                                    stateCopy,
                                    evaluator)));
        }
        return moveBewertungen.stream().max((p1, p2) -> p1.getEvaluation().compareTo(p2.getEvaluation()))
                .map(MoveEvaluationPair::getMove);
    }

    /**
     * The maximum Score of the maximizing player.
     *
     * @param depth         The depth of search tree.
     * @param currentPlayer The player who has to do the next move.
     * @param otherPlayer   The other player who has done the last move.
     * @param generator     The generator that creates next moves.
     * @param state         The current state.
     * @param evaluator     The abstract evaluator.
     * @return The maximizing score that represents the best move of maximizing player.
     * @throws GameException
     */
    public Double negamaxScore(final int depth, final P currentPlayer, final P otherPlayer,
            final NextMoveGenerator<P, S, M> generator, final S state, final Evaluation<P, S> evaluator)
            throws GameException {
        final Collection<M> allNextMoves = generator.getAllNextMoves(state, currentPlayer);
        if (depth == 0 || allNextMoves.isEmpty()) {
            return evaluator.evaluate(currentPlayer, state);
        }

        Double bestScore = Double.NEGATIVE_INFINITY;
        final Iterator<M> it = allNextMoves.iterator();
        while (it.hasNext()) {
            final M currentMove = it.next();
            final S stateCopy = state.deepCopy();
            currentMove.applyTo(stateCopy, currentPlayer);
            stateCopy.nextTurn();
            final Double val = -this
                    .negamaxScore(depth - 1, otherPlayer, currentPlayer, generator, stateCopy, evaluator);
            if (val > bestScore) {
                bestScore = val;
            }
        }
        return bestScore;
    }
}
