package de.fhdw.ipspiel21.viergewinnt.strategies.group1.strategy;

import java.util.ArrayList;
import java.util.List;

import de.fhdw.gaming.ipspiel21.searchtrees.domain.Evaluation;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntBoard;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntField;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntFieldState;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntPlayer;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntPosition;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntState;

/**
 * Evaluation for a Vier-Gewinnt Game. It compute all possible winning chances and evaluate the points for each player.
 * 
 */
public class VierGewinntEvaluation implements Evaluation<VierGewinntPlayer, VierGewinntState> {

    @Override
    public Double evaluate(final VierGewinntPlayer player, final VierGewinntState state) {
        final double currentPLayerSum;
        final double oppositePlayerSum;

        currentPLayerSum = evaluateHelper(buildWinningChances(state.getBoard(), player));
        oppositePlayerSum =
                evaluateHelper(buildWinningChances(state.getBoard(), this.getOtherPlayer(player, state)));

        return currentPLayerSum - oppositePlayerSum;
    }

    /**
     * Evalute the points of given winning chances (list of list with four items).
     *
     * see also buildWinningChances
     *
     * @param winningChances
     * @return playersum for player
     */
    private Double evaluateHelper(final List<List<VierGewinntField>> winningChances) {
        Double playersum = Double.valueOf(0);
        for (final List<VierGewinntField> currentWinningChance : winningChances) {
            int marksInWinningChance = 0;
            for (final VierGewinntField currentField : currentWinningChance) {
                if (!(currentField.getState().equals(VierGewinntFieldState.EMPTY))) {
                    marksInWinningChance++;
                }
            }
            switch (marksInWinningChance) {
            case 1:
                playersum = playersum + 1;
                break;
            case 2:
                playersum = playersum + 10;
                break;
            case 3:
                playersum = playersum + 50;
                break;
            case 4:
                playersum = Double.valueOf(1000);
                break;
            default:
                break;
            }
        }
        return playersum;
    }

    /**
     * Build all 78 winningChances for given player. Throw all not winnable chances away.
     * 
     * @param board  the actual vier-gewinnt board.
     * @param player the player which need his score of the board.
     * @return playersum
     */
    public List<List<VierGewinntField>> buildWinningChances(final VierGewinntBoard board,
            final VierGewinntPlayer player) {
        final List<List<VierGewinntField>> allAvailableWinningChances = new ArrayList<>();
        final List<List<VierGewinntField>> rows = getRows(board);
        for (final List<VierGewinntField> currentRow : rows) {
            allAvailableWinningChances.addAll(getWinningChancesfor(currentRow, player));
        }
        final List<List<VierGewinntField>> columns = getColumns(board);
        for (final List<VierGewinntField> currentColumn : columns) {
            allAvailableWinningChances.addAll(getWinningChancesfor(currentColumn, player));
        }
        final List<List<VierGewinntField>> diagonalLtoR = getDiagonalLtoR(board);
        for (final List<VierGewinntField> currentDiagonalLtoR : diagonalLtoR) {
            allAvailableWinningChances.addAll(getWinningChancesfor(currentDiagonalLtoR, player));
        }
        final List<List<VierGewinntField>> diagonalRtoL = getDiagonalRtoL(board);
        for (final List<VierGewinntField> currentDiagonalRtoL : diagonalRtoL) {
            allAvailableWinningChances.addAll(getWinningChancesfor(currentDiagonalRtoL, player));
        }
        return allAvailableWinningChances;
    }

    /**
     * Return all winningChances for a given row.
     * 
     * @param row    : given row
     * @param player : current player
     */
    private List<List<VierGewinntField>> getWinningChancesfor(final List<VierGewinntField> row,
            final VierGewinntPlayer player) {
        final VierGewinntFieldState oppositeState;
        if (player.getPlayerMark().equals(VierGewinntFieldState.YELLOW)) {
            oppositeState = VierGewinntFieldState.RED;
        } else {
            oppositeState = VierGewinntFieldState.YELLOW;
        }
        final List<List<VierGewinntField>> list = new ArrayList<>();
        for (int i = 0; i < row.size() - 3; i++) {
            final List<VierGewinntField> fields = new ArrayList<>();
            if (!row.get(i).getState().equals(oppositeState) && !row.get(i + 1).getState().equals(oppositeState)
                    && !row.get(i + 2).getState().equals(oppositeState)
                    && !row.get(i + 3).getState().equals(oppositeState)) {
                fields.add(row.get(i));
                fields.add(row.get(i + 1));
                fields.add(row.get(i + 2));
                fields.add(row.get(i + 3));
                list.add(fields);
            }
        }
        return list;
    }

    /**
     * Return all columns of the actual baord.
     * 
     * @param board : actual game board.
     * @return list of columns
     */
    private List<List<VierGewinntField>> getColumns(final VierGewinntBoard board) {
        final List<List<VierGewinntField>> listOfColumns = new ArrayList<>();
        for (int i = 0; i < board.getRowSize(); i++) {
            final List<VierGewinntField> columnlist = new ArrayList<>();
            for (int j = 0; j < board.getColumnSize(); j++) {
                columnlist.add(board.getFieldAt(VierGewinntPosition.of(i, j)));
            }
            listOfColumns.add(columnlist);
        }
        return listOfColumns;
    }

    /**
     * /** Return all rows of the actual baord.
     * 
     * @param board : actual game board.
     * @return list of rows
     */
    private List<List<VierGewinntField>> getRows(final VierGewinntBoard board) {
        final List<List<VierGewinntField>> listOfRows = new ArrayList<>();
        for (int i = 0; i < board.getColumnSize(); i++) {
            final List<VierGewinntField> rowlist = new ArrayList<>();
            for (int j = 0; j < board.getRowSize(); j++) {
                rowlist.add(board.getFieldAt(VierGewinntPosition.of(j, i)));
            }
            listOfRows.add(rowlist);
        }
        return listOfRows;
    }

    /**
     * Return all left to right down diagonals of the actual baord.
     * 
     * @param board : actual game board.
     * @return left to right down diagonals
     */
    public List<List<VierGewinntField>> getDiagonalLtoR(final VierGewinntBoard board) {
        final List<List<VierGewinntField>> listOfDiagonalsLtoR = new ArrayList<>();
        final int rowSize = board.getRowSize();
        final int columnSize = board.getColumnSize();
        for (int s = 0; s < rowSize; s++) {
            final List<VierGewinntField> diagonalLtoR = new ArrayList<>();
            for (int i = s; i > -1; i--) {
                diagonalLtoR.add(board.getFieldAt(VierGewinntPosition.of(s - i, i)));
            }
            listOfDiagonalsLtoR.add(diagonalLtoR);
        }

        for (int s = 1; s < columnSize; s++) {
            final List<VierGewinntField> diagonalLtoR = new ArrayList<>();
            for (int i = columnSize - 1; i > s; i--) {
                diagonalLtoR.add(board.getFieldAt(VierGewinntPosition.of(s + columnSize - 1 - i, i)));
            }
            listOfDiagonalsLtoR.add(diagonalLtoR);
        }
        return listOfDiagonalsLtoR;
    }

    /**
     * Return all right to left down diagonals of the actual baord.
     * 
     * @param board : actual game board.
     * @return right to left down diagonals
     */
    public List<List<VierGewinntField>> getDiagonalRtoL(final VierGewinntBoard board) {
        final List<List<VierGewinntField>> listOfDiagonalsRtoL = new ArrayList<>();
        final int rowSize = board.getRowSize();
        final int columnSize = board.getColumnSize();
        for (int s = 0; s < rowSize; s++) {
            final List<VierGewinntField> diagonalRtoL = new ArrayList<>();
            for (int i = s; i > -1; i--) {
                diagonalRtoL.add(board.getFieldAt(VierGewinntPosition.of(i, (columnSize - 1) - s + i)));
            }
            listOfDiagonalsRtoL.add(diagonalRtoL);
        }

        for (int s = 1; s < columnSize; s++) {
            final List<VierGewinntField> diagonalRtoL = new ArrayList<>();
            for (int i = s - 1; i < rowSize; i++) {
                diagonalRtoL.add(board.getFieldAt(VierGewinntPosition.of(i, i - s + 1)));
            }
            listOfDiagonalsRtoL.add(diagonalRtoL);
        }
        return listOfDiagonalsRtoL;
    }

    /**
     * Helper method to receive the opponent player.
     * 
     * @param player : current player
     * @param state  : current state
     */
    private VierGewinntPlayer getOtherPlayer(final VierGewinntPlayer player, final VierGewinntState state) {
        if (player.isUsingYellow()) {
            return state.getRedPlayer();
        } else {
            return state.getYellowPlayer();
        }
    }
}
