/*
 * Copyright © 2021 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel21-Dilemma.
 *
 * Ipspiel21-Dil is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel21-Dil is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel21-Dilemma. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.impl;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaPlayer;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaPlayerBuilder;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.domain.DilemmaState;
import de.fhdw.gaming.ipspiel21.dilemmaOriginal.moves.impl.AbstractDilemmaMove;

/**
 * Implements {@link DilemmaState}.
 */
final class DilemmaStateImpl implements DilemmaState {

    /**
     * The first player.
     */
    private final DilemmaPlayer firstPlayer;
    /**
     * The second player.
     */
    private final DilemmaPlayer secondPlayer;
    /**
     * The states of the players.
     **/
    private final Map<String, PlayerState> playerStates;
    /**
     * This map contains the outcomes of the current Player.
     */
    private final Map<String, Double> playerOutcomes;
    /**
     * This map contains the Matching between Outcome and PlayerStates.
     */
    private final Map<Double, PlayerState> matchPlayerstate;

    /**
     * Creates a Dilemma state.
     *
     * @param firstPlayerBuilder  A builder for the first player.
     * @param secondPlayerBuilder A builder for the second player.
     * @throws GameException if the state cannot be created according to the rules of the game.
     */
    DilemmaStateImpl(final DilemmaPlayerBuilder firstPlayerBuilder, final DilemmaPlayerBuilder secondPlayerBuilder)
            throws GameException {

        this.firstPlayer = Objects.requireNonNull(firstPlayerBuilder, "firstPlayerBuilder").build(this);
        this.secondPlayer = Objects.requireNonNull(secondPlayerBuilder, "secondPlayerBuilder").build(this);

        this.playerOutcomes = new LinkedHashMap<>();

        this.playerStates = new LinkedHashMap<>();
        this.playerStates.put(this.firstPlayer.getName(), PlayerState.PLAYING);
        this.playerStates.put(this.secondPlayer.getName(), PlayerState.PLAYING);

        this.matchPlayerstate = new LinkedHashMap<>();
        this.matchPlayerstate.put(0.00, PlayerState.WON);
        this.matchPlayerstate.put(-1.00, PlayerState.DRAW);
        this.matchPlayerstate.put(-10.00, PlayerState.LOST);
        this.matchPlayerstate.put(-8.00, PlayerState.DRAW);

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                    String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Creates a Dilemma state by copying an existing one.
     *
     * @param source The state to copy.
     */
    DilemmaStateImpl(final DilemmaStateImpl source) {
        this.firstPlayer = source.firstPlayer.deepCopy(this);
        this.secondPlayer = source.secondPlayer.deepCopy(this);
        this.playerOutcomes = new LinkedHashMap<>(source.playerOutcomes);
        this.playerStates = new LinkedHashMap<>();
        this.playerStates.put(this.firstPlayer.getName(), source.playerStates.get(this.firstPlayer.getName()));
        this.playerStates.put(this.secondPlayer.getName(), source.playerStates.get(this.secondPlayer.getName()));
        this.matchPlayerstate = source.matchPlayerstate;
    }

    /**
     * Returns the first player.
     */
    @Override
    public DilemmaPlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Returns the second player.
     */
    @Override
    public DilemmaPlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public String toString() {
        return String.format("DilemmaState[firstPlayer=%s, secondPlayer=%s, playerOutcomes=%s, playerStates=%s]",
                this.firstPlayer, this.secondPlayer, this.playerOutcomes, this.playerStates);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof DilemmaStateImpl) {
            final DilemmaStateImpl other = (DilemmaStateImpl) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public DilemmaState deepCopy() {
        return this;

    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    @Override
    public Map<String, DilemmaPlayer> getPlayers() {
        final Map<String, DilemmaPlayer> result = new LinkedHashMap<>();
        result.put(this.firstPlayer.getName(), this.firstPlayer);
        result.put(this.secondPlayer.getName(), this.secondPlayer);
        return result;
    }

    @Override
    public PlayerState getPlayerState(final String playerName) throws IllegalArgumentException {
        final PlayerState playerState = this.playerStates.get(playerName);
        if (playerState != null) {
            return playerState;
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
    }

    @Override
    public void setPlayerState(final String playerName, final PlayerState newState)
            throws IllegalArgumentException {
        if (this.playerStates.containsKey(playerName)) {
            this.playerStates.put(playerName, newState);
            if (newState.equals(PlayerState.PLAYING)) {
                this.playerOutcomes.remove(playerName);
            }
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
    }

    @Override
    public Optional<Double> getPlayerOutcome(final String playerName) throws IllegalArgumentException {

        if (this.playerStates.containsKey(playerName)) {
            final Double outcome = this.playerOutcomes.get(playerName);
            return outcome != null ? Optional.of(outcome) : DilemmaState.super.getPlayerOutcome(playerName);
        } else {
            throw new IllegalArgumentException(String.format("Unknown player %s.", playerName));
        }
    }

    @Override
    public void setPlayerOutcome(final String playerName, final double newOutcome)
            throws IllegalArgumentException {
        if (this.getPlayerState(playerName).equals(PlayerState.PLAYING)) {
            throw new IllegalArgumentException(String.format("Cannot set outcome for player %s.", playerName));
        } else {
            this.playerOutcomes.put(playerName, newOutcome);
        }
    }

    @Override
    public Set<DilemmaPlayer> computeNextPlayers() {
        final Set<DilemmaPlayer> playersWithoutAnswer = new LinkedHashSet<>();
        if (this.getFirstPlayer().getMove() == null) {
            playersWithoutAnswer.add(firstPlayer);
        }
        if (this.getSecondPlayer().getMove() == null) {
            playersWithoutAnswer.add(secondPlayer);
        }
        return playersWithoutAnswer;
    }

    @Override
    public void nextTurn() {
        calculateOutcome();

    }

    /**
     * this function calculate the Outcome after all(here 2) player took their move. it uses the help function
     * generateOuctomes().
     */
    private void calculateOutcome() {
        if (this.computeNextPlayers().isEmpty()) {
            generateOutcomes(this.firstPlayer, this.secondPlayer);
            generateOutcomes(this.secondPlayer, this.firstPlayer);
        }
    }

    /**
     * Generate the Outcome for the first player.
     * 
     * Here we go over the possibleOutcomes of the first player and generate the double from the second decision.
     * 
     * @param fromP1 Player1 for the Outcome.
     * @param p2     Player2 for the Outcome.
     */
    private void generateOutcomes(final DilemmaPlayer fromP1, final DilemmaPlayer p2) {
        final Map<AbstractDilemmaMove, Map<AbstractDilemmaMove, Double>> matches =
                this.firstPlayer.getPossibleOutcomes();
        final Map<AbstractDilemmaMove, Double> forplayer1 = matches.get(fromP1.getMove());
        final Double forPlayer2 = forplayer1.get(p2.getMove());
        this.setPlayerState(fromP1.getName(), this.matchPlayerstate.get(forPlayer2));
        this.setPlayerOutcome(fromP1.getName(), forPlayer2);
    }

    @Override
    public void setAnswer(final DilemmaPlayer player, final AbstractDilemmaMove move) {
        this.getPlayers().get(player.getName()).setMove(move);
    }

    @Override
    public DilemmaPlayer getOtherPlayer(final DilemmaPlayer player) {
        return this.firstPlayer.equals(player) ? this.secondPlayer : this.firstPlayer;
    }

}
