
package de.fhdw.gaming.ipspiel21.viergewinnt.gui.impl;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.IntStream;

import de.fhdw.gaming.core.domain.Game;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.core.domain.Player;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.core.domain.State;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntBoard;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntFieldState;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntGame;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntObserver;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntPlayer;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntPosition;
import de.fhdw.gaming.ipspiel21.viergewinnt.core.domain.VierGewinntState;
import javafx.animation.Animation;
import javafx.animation.KeyFrame;
import javafx.animation.Timeline;
import javafx.application.Platform;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.beans.value.ObservableValue;
import javafx.geometry.HPos;
import javafx.geometry.Insets;
import javafx.geometry.Pos;
import javafx.geometry.VPos;
import javafx.scene.control.Label;
import javafx.scene.control.Slider;
import javafx.scene.layout.Background;
import javafx.scene.layout.BackgroundFill;
import javafx.scene.layout.Border;
import javafx.scene.layout.BorderStroke;
import javafx.scene.layout.BorderStrokeStyle;
import javafx.scene.layout.ColumnConstraints;
import javafx.scene.layout.CornerRadii;
import javafx.scene.layout.GridPane;
import javafx.scene.layout.HBox;
import javafx.scene.layout.Priority;
import javafx.scene.layout.RowConstraints;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;
import javafx.scene.text.Font;
import javafx.scene.text.FontPosture;
import javafx.scene.text.FontWeight;
import javafx.stage.Screen;
import javafx.util.Duration;

/**
 * Displays an VierGewinnt board.
 */
@SuppressWarnings("PMD.TooManyFields")
final class VierGewinntBoardView implements VierGewinntObserver {

    /**
     * The initial delay in seconds.
     */
    private static final double INITIAL_DELAY = 0.5;
    /**
     * The margin used by elements of the state pane.
     */
    private static final double STATE_PANE_MARGIN = 40.0;
    /**
     * Pattern for extracting the relevant parts of a strategy name.
     */
    private static final Pattern STRATEGY_NAME_PATTERN = Pattern.compile("^(VierGewinnt)?(.*?)(Strategy)?$");
    /**
     * The factor to multiply pixels with to receive points.
     */
    private final double pixelsToPointsFactor;
    /**
     * The field controls.
     */
    private final Map<VierGewinntPosition, VierGewinntFieldView> controls;
    /**
     * The pane containing everything.
     */
    private final HBox rootPane;
    /**
     * The pane containing the board and its border.
     */
    private final VBox boardPane;
    /**
     * The grid pane containing the fields.
     */
    private final GridPane fieldPane;
    /**
     * The label containing the number of yellow marks on the board.
     */
    private Label yellowMarks;
    /**
     * The label containing the number of red marks on the board.
     */
    private Label redMarks;
    /**
     * The label describing the current game state.
     */
    private Label gameStateDescription;
    /**
     * The animation of the current game state description (if any).
     */
    private Optional<Timeline> gameStateDescriptionAnimation;
    /**
     * The delay in milliseconds.
     */
    private final AtomicInteger delay;
    /**
     * The size of a field control on basis of a row.
     */
    private final SimpleDoubleProperty fieldControlSize;
    /**
     * The size of various margins.
     */
    private final SimpleDoubleProperty margin;
    /**
     * The padding used for GridPanes.
     */
    private final SimpleObjectProperty<Insets> gridPadding;
    /**
     * The font size in pixels.
     */
    private final SimpleDoubleProperty fontSizeInPixels;
    /**
     * The font used for border labels.
     */
    private final SimpleObjectProperty<Font> borderLabelFont;
    /**
     * The font used for labels texts in the player's pane.
     */
    private final SimpleObjectProperty<Font> labelTextFont;
    /**
     * The font used for label values in the player's pane.
     */
    private final SimpleObjectProperty<Font> labelValueFont;
    /**
     * The font used for displaying the game result.
     */
    private final SimpleObjectProperty<Font> gameResultFont;
    /**
     * The pane for the top edge.
     */
    private HBox topEdge;
    /**
     * The pane for the left edge.
     */
    private VBox leftEdge;
    /**
     * The pane for the right edge.
     */
    private VBox rightEdge;
    /**
     * The pane for the bottom edge.
     */
    private HBox bottomEdge;
    /**
     * The last game state.
     */
    private Optional<VierGewinntState> lastGameState;
    /**
     * The {@link Semaphore} used by {@link VierGewinntBoardEventProviderImpl}.
     */
    private final Semaphore semaphore = new Semaphore(0);

    /**
     * Creates an {@link VierGewinntBoardView}.
     *
     * @param game The game.
     */
    VierGewinntBoardView(final VierGewinntGame game) {
        this.pixelsToPointsFactor = 36.0 / Screen.getPrimary().getDpi();
        this.borderLabelFont = new SimpleObjectProperty<>(Font.getDefault());
        this.labelTextFont = new SimpleObjectProperty<>(Font.getDefault());
        this.labelValueFont = new SimpleObjectProperty<>(Font.getDefault());
        this.gameResultFont = new SimpleObjectProperty<>(Font.getDefault());

        this.fieldControlSize = new SimpleDoubleProperty(0.0);
        this.margin = new SimpleDoubleProperty(0.0);
        this.gridPadding = new SimpleObjectProperty<>(Insets.EMPTY);
        this.fontSizeInPixels = new SimpleDoubleProperty(0.0);

        this.controls = new LinkedHashMap<>();
        this.fieldPane = new GridPane();
        this.boardPane = this.createFieldWithBorderPane();

        final GridPane statePane = this.createStatePane(game);

        this.rootPane = new HBox();
        this.rootPane.getChildren().addAll(this.boardPane, statePane);
        HBox.setHgrow(this.boardPane, Priority.ALWAYS);
        HBox.setHgrow(statePane, Priority.ALWAYS);
        HBox.setMargin(statePane, new Insets(VierGewinntBoardView.STATE_PANE_MARGIN));

        this.delay = new AtomicInteger((int) (VierGewinntBoardView.INITIAL_DELAY * 1000.0));
        this.lastGameState = Optional.empty();
        game.addObserver(this);

        this.gameStateDescriptionAnimation = Optional.empty();
    }

    /**
     * Creates the pane displaying the board and its border.
     * <p>
     * Requires {@link #fieldControlSize} to be valid.
     */
    private VBox createFieldWithBorderPane() {
        final Background background = new Background(
                new BackgroundFill(Color.WHITE, CornerRadii.EMPTY, Insets.EMPTY));

        final HBox topLeftCorner = new HBox();
        topLeftCorner.setBackground(background);
        topLeftCorner.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        topLeftCorner.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));
        this.topEdge = new HBox();
        this.topEdge.setBackground(background);
        this.topEdge.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        this.topEdge.setAlignment(Pos.CENTER);
        final HBox topRightCorner = new HBox();
        topRightCorner.setBackground(background);
        topRightCorner.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        topRightCorner.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));

        HBox.setHgrow(topLeftCorner, Priority.NEVER);
        HBox.setHgrow(this.topEdge, Priority.NEVER);
        HBox.setHgrow(topRightCorner, Priority.NEVER);

        this.leftEdge = new VBox();
        this.leftEdge.setBackground(background);
        this.leftEdge.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));
        this.leftEdge.setAlignment(Pos.CENTER);
        this.rightEdge = new VBox();
        this.rightEdge.setBackground(background);
        this.rightEdge.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));
        this.rightEdge.setAlignment(Pos.CENTER);

        HBox.setHgrow(this.leftEdge, Priority.NEVER);
        HBox.setHgrow(this.fieldPane, Priority.NEVER);
        HBox.setHgrow(this.rightEdge, Priority.NEVER);

        final HBox bottomLeftCorner = new HBox();
        bottomLeftCorner.setBackground(background);
        bottomLeftCorner.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        bottomLeftCorner.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));
        this.bottomEdge = new HBox();
        this.bottomEdge.setBackground(background);
        this.bottomEdge.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        this.bottomEdge.setAlignment(Pos.CENTER);
        final HBox bottomRightCorner = new HBox();
        bottomRightCorner.setBackground(background);
        bottomRightCorner.minHeightProperty().bind(this.fieldControlSize.multiply(0.5));
        bottomRightCorner.minWidthProperty().bind(this.fieldControlSize.multiply(0.5));

        HBox.setHgrow(bottomLeftCorner, Priority.NEVER);
        HBox.setHgrow(this.bottomEdge, Priority.NEVER);
        HBox.setHgrow(bottomRightCorner, Priority.NEVER);

        final HBox top = new HBox();
        top.getChildren().addAll(topLeftCorner, this.topEdge, topRightCorner);
        final HBox centre = new HBox();
        centre.getChildren().addAll(this.leftEdge, this.fieldPane, this.rightEdge);
        final HBox bottom = new HBox();
        bottom.getChildren().addAll(bottomLeftCorner, this.bottomEdge, bottomRightCorner);

        VBox.setVgrow(top, Priority.NEVER);
        VBox.setVgrow(centre, Priority.NEVER);
        VBox.setVgrow(bottom, Priority.NEVER);

        final VBox vbox = new VBox();
        vbox.getChildren().addAll(top, centre, bottom);

        return vbox;
    }

    /**
     * Creates the pane displaying the game state.
     *
     * @param game The game.
     */
    private GridPane createStatePane(final VierGewinntGame game) {
        final GridPane yellowMarksPane = this.createPlayerPane(game, game.getState().getYellowPlayer());
        final GridPane redMarksPane = this.createPlayerPane(game, game.getState().getRedPlayer());

        // final Font font = Font.font(null, FontWeight.BOLD, FontPosture.REGULAR, 40.0);
        this.gameStateDescription = new Label();
        this.gameStateDescription.fontProperty().bind(this.gameResultFont);
        this.gameStateDescription.setTextFill(Color.RED);

        final Label delayLabel = new Label("Delay in seconds: ");

        final Slider delaySlider = new Slider(0.0, 5.0, VierGewinntBoardView.INITIAL_DELAY);
        delaySlider.setMajorTickUnit(VierGewinntBoardView.INITIAL_DELAY);
        delaySlider.setMinorTickCount(4);
        delaySlider.setBlockIncrement(VierGewinntBoardView.INITIAL_DELAY);
        delaySlider.setShowTickMarks(true);
        delaySlider.setShowTickLabels(true);
        delaySlider.snapToTicksProperty().set(true);
        delaySlider.valueProperty().addListener(
                (final ObservableValue<? extends Number> observable, final Number oldValue, final Number newValue) -> {
                    this.delay.set((int) (newValue.doubleValue() * 1000.0));
                });

        HBox.setHgrow(delayLabel, Priority.NEVER);
        HBox.setHgrow(delaySlider, Priority.ALWAYS);

        final GridPane gridPane = new GridPane();
        gridPane.vgapProperty().bind(this.margin.multiply(2.0));

        gridPane.addRow(0, yellowMarksPane);
        GridPane.setHalignment(yellowMarksPane, HPos.CENTER);
        GridPane.setHgrow(yellowMarksPane, Priority.ALWAYS);
        GridPane.setVgrow(yellowMarksPane, Priority.NEVER);

        gridPane.addRow(1, redMarksPane);
        GridPane.setHalignment(redMarksPane, HPos.CENTER);
        GridPane.setHgrow(redMarksPane, Priority.ALWAYS);
        GridPane.setVgrow(redMarksPane, Priority.NEVER);

        gridPane.addRow(2, this.gameStateDescription);
        GridPane.setHalignment(this.gameStateDescription, HPos.CENTER);
        GridPane.setHgrow(this.gameStateDescription, Priority.ALWAYS);
        GridPane.setVgrow(this.gameStateDescription, Priority.NEVER);

        return gridPane;
    }

    /**
     * Returns the root pane.
     */
    HBox getNode() {
        return this.rootPane;
    }

    /**
     * Returns the {@link Semaphore} used for user interaction.
     */
    Semaphore getUserInputSemaphore() {
        return this.semaphore;
    }

    /**
     * Maps a position to the corresponding {@link VierGewinntFieldView}.
     *
     * @param position The position.
     * @return The field view (if it exists).
     */
    Optional<VierGewinntFieldView> getFieldView(final VierGewinntPosition position) {
        return Optional.ofNullable(this.controls.get(position));
    }

    /**
     * Sets the game state.
     *
     * @param text The game state.
     */
    private void setGameState(final String text) {
        this.gameStateDescriptionAnimation.ifPresent((final Timeline timeline) -> timeline.stop());
        this.gameStateDescription.setText(text);
        if (!text.isEmpty()) {
            this.gameStateDescriptionAnimation = Optional.of(
                    new Timeline(
                            new KeyFrame(Duration.seconds(0.5), evt -> this.gameStateDescription.setVisible(false)),
                            new KeyFrame(Duration.seconds(1.0), evt -> this.gameStateDescription.setVisible(true))));
            this.gameStateDescriptionAnimation.get().setCycleCount(Animation.INDEFINITE);
            this.gameStateDescriptionAnimation.get().play();
        }
    }

    /**
     * Called if the game has been paused.
     *
     * @param game The game.
     */
    void gamePaused(final VierGewinntGame game) {
        Platform.runLater(() -> this.setGameState("P A U S E D"));
    }

    /**
     * Called if the game has been resumed.
     *
     * @param game The game.
     */
    void gameResumed(final VierGewinntGame game) {
        Platform.runLater(() -> this.setGameState(""));
    }

    @Override
    public void started(final Game<?, ?, ?, ?> game, final State<?, ?> state) throws InterruptedException {
        Platform.runLater(() -> {
            this.lastGameState = Optional.of((VierGewinntState) state);

            final VierGewinntBoard board = this.lastGameState.get().getBoard();
            final int numRows = board.getRowSize();
            final int numColumns = board.getColumnSize();
            
            this.addLabelsToEdge(numColumns);

            this.boardPane.widthProperty().addListener(
                    (final ObservableValue<? extends Number> observable, final Number oldValue,
                            final Number newValue) -> {
                        final double size = Math.min(newValue.doubleValue(), this.boardPane.getHeight());
                        this.setBorderAndFieldSize(numColumns, size);
                    });
            this.boardPane.heightProperty().addListener(
                    (final ObservableValue<? extends Number> observable, final Number oldValue,
                            final Number newValue) -> {
                        final double size = Math.min(newValue.doubleValue(), this.boardPane.getWidth());
                        this.setBorderAndFieldSize(numRows, size);
                    });

            this.fieldPane.widthProperty().addListener(
                    (final ObservableValue<? extends Number> observable, final Number oldValue,
                            final Number newValue) -> {
                        final double size = Math.max(newValue.doubleValue(), this.fieldPane.getHeight());
                        this.fieldControlSize.set(size / numColumns);
                    });
            this.fieldPane.heightProperty().addListener(
                    (final ObservableValue<? extends Number> observable, final Number oldValue,
                            final Number newValue) -> {
                        final double size = Math.max(newValue.doubleValue(), this.fieldPane.getWidth());
                        this.fieldControlSize.set(size / numRows);
                    });

            IntStream.range(0, numColumns).forEachOrdered((final int column) -> {
                IntStream.range(0, numRows).forEachOrdered((final int row) -> {
                    final VierGewinntPosition position = VierGewinntPosition.of(row, column);
                    final VierGewinntFieldState fieldState = board.getFieldAt(position).getState();
                    final VierGewinntFieldView fieldControl = new VierGewinntFieldView(fieldState);
                    this.fieldPane.add(fieldControl, column, row);
                    this.controls.put(position, fieldControl);

                    fieldControl.prefWidthProperty().bind(this.fieldControlSize);
                    fieldControl.prefHeightProperty().bind(this.fieldControlSize);
                });
            });
        });

        this.delayNextMove();
    }

    /**
     * Adds the labels to the top and left edges.
     *
     * @param numColumns The number of rows  of the board.
     */
    private void addLabelsToEdge(final int numColumns) {
         this.borderLabelFont.set(Font.font(null, FontWeight.NORMAL, FontPosture.REGULAR, 100.00));
         for (char i = 0; i < numColumns; ++i) {
            final Label topLabel = new Label("\u25BC");
            topLabel.fontProperty().bind(this.borderLabelFont);
            topLabel.setMaxWidth(Double.MAX_VALUE);
            topLabel.setAlignment(Pos.CENTER);
            this.topEdge.getChildren().add(topLabel);
            HBox.setHgrow(topLabel, Priority.ALWAYS);
        }
    }

    /**
     * Sets the size of the board's fields and border.
     *
     * @param numRowsAndColumns The number of rows (and columns) of the board.
     * @param boardSizeInPixels The width (and height) of the board in pixels.
     */
    private void setBorderAndFieldSize(final int numRowsAndColumns, final double boardSizeInPixels) {
        final double boardSizeWithoutBorders = boardSizeInPixels * numRowsAndColumns / (numRowsAndColumns + 1);
        this.fieldPane.setPrefSize(boardSizeWithoutBorders, boardSizeWithoutBorders);
        this.topEdge.setPrefWidth(boardSizeWithoutBorders);
        this.leftEdge.setPrefHeight(boardSizeWithoutBorders);
        this.rightEdge.setPrefHeight(boardSizeWithoutBorders);
        this.bottomEdge.setPrefWidth(boardSizeWithoutBorders);

        this.fontSizeInPixels.set(this.fieldControlSize.get() * 0.5);
        final double fontSize = this.fontSizeInPixels.get() * this.pixelsToPointsFactor;

        final Font fontRegular = Font.font(null, FontWeight.NORMAL, FontPosture.REGULAR, fontSize);
        final Font fontBold = Font.font(null, FontWeight.BOLD, FontPosture.REGULAR, fontSize);

        this.borderLabelFont.set(fontRegular);
        this.labelTextFont.set(fontBold);
        this.labelValueFont.set(fontRegular);
        this.gameResultFont.set(fontBold);

        this.margin.set(this.fontSizeInPixels.get() * 0.25);
        this.gridPadding.set(new Insets(this.margin.get()));
    }

    @Override
    public void nextPlayersComputed(final Game<?, ?, ?, ?> game, final State<?, ?> state,
            final Set<? extends Player> players) {
        // nothing to do
    }

    @Override
    public void illegalPlayerRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player player) {
        // nothing to do
    }

    @Override
    public void legalMoveApplied(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player player,
            final Move<?, ?> move) throws InterruptedException {

        Platform.runLater(() -> {
            final VierGewinntState vierGewinntState = (VierGewinntState) state;

            this.updateFields(vierGewinntState, VierGewinntFieldState.YELLOW, this.yellowMarks);
            this.updateFields(vierGewinntState, VierGewinntFieldState.RED, this.redMarks);

            this.lastGameState = Optional.of(vierGewinntState);
        });

        this.delayNextMove();
    }

    /**
     * Updates the fields and the label for a given field state after a move.
     *
     * @param vierGewinntState The game state.
     * @param fieldState     The state of the fields to update ({@link VierGewinntFieldState#BLACK} or
     *                       {@link VierGewinntFieldState#WHITE}).
     * @param label          The label to update ({@link #yellowMarks} or {@link #redMarks}).
     */
    private void updateFields(final VierGewinntState vierGewinntState, final VierGewinntFieldState fieldState,
            final Label label) {
        final Set<VierGewinntPosition> currentFieldsInDesiredState = new LinkedHashSet<>(
                vierGewinntState.getBoard().getFieldsBeing(fieldState).keySet());
        label.setText(Integer.toString(currentFieldsInDesiredState.size()));
        currentFieldsInDesiredState.removeAll(this.lastGameState.get().getBoard().getFieldsBeing(fieldState).keySet());
        currentFieldsInDesiredState
                .forEach((final VierGewinntPosition position) -> this.controls.get(position).setFieldState(fieldState));
    }

    @Override
    public void illegalMoveRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player player,
            final Optional<Move<?, ?>> move, final String reason) {
        // nothing to do
    }

    @Override
    public void overdueMoveRejected(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player player,
            final Optional<Move<?, ?>> chosenMove) {
        // nothing to do
    }

    @Override
    public void playerResigned(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player player) {
        // nothing to do
    }

    @Override
    public void playerOvertaken(final Game<?, ?, ?, ?> game, final State<?, ?> state, final Player overtakenPlayer,
            final Player overtakingPlayer) {
        // nothing to do
    }

    @Override
    public void finished(final Game<?, ?, ?, ?> game, final State<?, ?> state) {
        Platform.runLater(() -> {
            final VierGewinntState vierGewinntState = (VierGewinntState) state;
            if (vierGewinntState.getYellowPlayer().getState().equals(PlayerState.WON)) {
                this.setGameState("YELLOW WINS!");
            } else if (vierGewinntState.getRedPlayer().getState().equals(PlayerState.WON)) {
                this.setGameState("RED WINS!");
            } else if (vierGewinntState.getYellowPlayer().getState().equals(PlayerState.DRAW)) {
                this.setGameState("DRAW GAME!");
            }
        });
    }

    /**
     * Destroys this object.
     *
     * @param game The associated game.
     */
    public void destroy(final VierGewinntGame game) {
        this.gameStateDescriptionAnimation.ifPresent((final Timeline timeline) -> timeline.stop());
        this.semaphore.release();
        game.removeObserver(this);
    }

    /**
     * Delays the execution of the next move.
     */
    private void delayNextMove() throws InterruptedException {
        Thread.sleep(this.delay.get());
    }

    /**
     * Returns a pane displaying the state of one player.
     *
     * @param game   The game.
     * @param player The player.
     * @return The requested pane.
     */
    private GridPane createPlayerPane(final VierGewinntGame game, final VierGewinntPlayer player) {
        final Label nameLabel = new Label("Name:");
        nameLabel.fontProperty().bind(this.labelTextFont);

        final Label nameText = new Label(player.getName());
        nameText.fontProperty().bind(this.labelValueFont);

        final Label markLabel = new Label("Color:");
        markLabel.fontProperty().bind(this.labelTextFont);

        final Label markText = new Label();
        markText.fontProperty().bind(this.labelValueFont);
        if (player.isUsingYellow()) {
            markText.setText("Yellow");
        } else {
            markText.setText("Red");
        }

        final Label strategyLabel = new Label("Strategy:");
        strategyLabel.fontProperty().bind(this.labelTextFont);

        final Label strategyText;
        final String rawName = game.getStrategies().get(player.getName()).toString();
        final Matcher nameMatcher = VierGewinntBoardView.STRATEGY_NAME_PATTERN.matcher(rawName);
        if (nameMatcher.matches()) {
            strategyText = new Label(nameMatcher.group(2));
        } else {
            strategyText = new Label(rawName);
        }
        strategyText.fontProperty().bind(this.labelValueFont);

        final Label tokensLabel = new Label("Tokens:");
        tokensLabel.fontProperty().bind(this.labelTextFont);

        final Label tokensText = new Label("0");
        tokensText.fontProperty().bind(this.labelValueFont);
        if (player.isUsingYellow()) {
            this.yellowMarks = tokensText;
        } else {
            this.redMarks = tokensText;
        }

        final GridPane gridPane = new GridPane();

        final ColumnConstraints labelColumnConstraints = new ColumnConstraints();
        labelColumnConstraints.setHalignment(HPos.RIGHT);
        labelColumnConstraints.setHgrow(Priority.NEVER);
        final ColumnConstraints textColumnConstraints = new ColumnConstraints();
        textColumnConstraints.setHalignment(HPos.LEFT);
        textColumnConstraints.setHgrow(Priority.ALWAYS);
        textColumnConstraints.setMaxWidth(Double.MAX_VALUE);
        gridPane.getColumnConstraints().addAll(labelColumnConstraints, textColumnConstraints);

        final RowConstraints rowConstraints = new RowConstraints();
        rowConstraints.setValignment(VPos.TOP);
        gridPane.getRowConstraints().addAll(rowConstraints, rowConstraints, rowConstraints, rowConstraints);

        gridPane.addRow(0, nameLabel, nameText);
        gridPane.addRow(1, markLabel, markText);
        gridPane.addRow(2, strategyLabel, strategyText);
        gridPane.addRow(3, tokensLabel, tokensText);

        gridPane.setBorder(
                new Border(
                        new BorderStroke(Color.GREY, BorderStrokeStyle.SOLID, 
                                CornerRadii.EMPTY, BorderStroke.THICK)));

        gridPane.hgapProperty().bind(this.margin.multiply(2.0));
        gridPane.paddingProperty().bind(this.gridPadding);

        return gridPane;
    }
}
