package de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteAnswerEnum;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteGameBuilder;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteGameBuilderFactory;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKantePlayer;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKantePlayerBuilder;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteStrategy;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.factory.KopfundZahlDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.factory.KopfundZahlStrategyFactory;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.factory.KopfundZahlStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.moves.factory.KopfundZahlundKanteMoveFactory;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.moves.impl.DefaultKopfundZahlundKanteMoveFactory;

/**
 * Implements {@link KopfundZahlundKanteGameBuilderFactory} by creating a KopfundZahl game builder.
 */
public final class KopfundZahlundKanteGameBuilderFactoryImpl implements KopfundZahlundKanteGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;

    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;

    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Head and Tail strategies.
     */
    private final Set<KopfundZahlundKanteStrategy> strategies;

    /**
     * Creates a Head and Tail game factory. Head and Tail strategies are loaded
     * by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public KopfundZahlundKanteGameBuilderFactoryImpl() {
        this(new KopfundZahlDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Head and Tail game factory.
     *
     * @param strategyFactoryProvider The {@link KopfundZahlStrategyFactoryProvider} for loading Head and
     *                                Tail strategies.
     */
    KopfundZahlundKanteGameBuilderFactoryImpl(final KopfundZahlDefaultStrategyFactoryProvider strategyFactoryProvider) {
        final KopfundZahlundKanteMoveFactory moveFactory = new DefaultKopfundZahlundKanteMoveFactory();

        final List<KopfundZahlStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final KopfundZahlStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "KopfundZahlundKante";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return KopfundZahlundKanteGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return KopfundZahlundKanteGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public KopfundZahlundKanteGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final KopfundZahlundKanteGameBuilder gameBuilder = new KopfundZahlundKanteGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String, Object> gameData = inputProvider.needInteger(
                    GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                    "Maximum computation time per move in seconds",
                    Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MinValueValidator<>(
                            KopfundZahlundKanteGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MaxValueValidator<>(
                            KopfundZahlundKanteGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                    .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1");
            final KopfundZahlundKantePlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(),
                    firstPlayerData);
            final KopfundZahlundKanteStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2");
            final KopfundZahlundKantePlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(),
                    secondPlayerData);
            final KopfundZahlundKanteStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating KopfundZahl game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {
        if (title.equals("Player 1")) {
            inputProvider
                    .needString(
                            GameBuilderFactory.PARAM_PLAYER_NAME,
                            "Name",
                            Optional.empty(),
                            new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_TAIL,
                            "Player's outcome on Tail/Tail",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_HEAD,
                            "Player's outcome on Tail/Head",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_TAIL,
                            "Player's outcome on Head/Tail",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_HEAD,
                            "Player's outcome on Head/Head",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_EDGE_HEAD,
                            "Player's outcome on Edge/Head",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_EDGE,
                            "Player's outcome on Head/Edge",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_EDGE_TAIL,
                            "Player's outcome on Edge/Tail",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_EDGE,
                            "Player's outcome on Tail/Edge",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_EDGE_EDGE,
                            "Player's outcome on Edge/Edge",
                            Optional.of(1))
                    .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(),
                            this.strategies);

        } else {
            inputProvider
                    .needString(
                            GameBuilderFactory.PARAM_PLAYER_NAME,
                            "Name",
                            Optional.empty(),
                            new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_TAIL,
                            "Player's outcome on Tail/Tail",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_HEAD,
                            "Player's outcome on Tail/Head",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_TAIL,
                            "Player's outcome on Head/Tail",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_HEAD,
                            "Player's outcome on Head/Head",
                            Optional.of(-1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_EDGE_HEAD,
                            "Player's outcome on Edge/Head",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_EDGE,
                            "Player's outcome on Head/Edge",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_EDGE_TAIL,
                            "Player's outcome on Edge/Tail",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_EDGE,
                            "Player's outcome on Tail/Edge",
                            Optional.of(1))
                    .needInteger(
                            KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_EDGE_EDGE,
                            "Player's outcome on Edge/Edge",
                            Optional.of(-1))
                    .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(),
                            this.strategies);

        }
        return inputProvider.requestData(title);
    }

    /**
     * Creates a KopfundZahl player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link KopfundZahlundKantePlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private KopfundZahlundKantePlayer createPlayer(final KopfundZahlundKantePlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<KopfundZahlundKanteAnswerEnum,
                Map<KopfundZahlundKanteAnswerEnum, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<KopfundZahlundKanteAnswerEnum, Double> possibleOutcomesTail = new LinkedHashMap<>();
        possibleOutcomesTail.put(
                KopfundZahlundKanteAnswerEnum.TAIL,
                (double) (Integer) playerData
                        .get(KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_TAIL));
        possibleOutcomesTail.put(
                KopfundZahlundKanteAnswerEnum.EDGE,
                (double) (Integer) playerData
                        .get(KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_EDGE));
        possibleOutcomesTail.put(
                KopfundZahlundKanteAnswerEnum.HEAD,
                (double) (Integer) playerData
                        .get(KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_TAIL_HEAD));
        possibleOutcomes.put(KopfundZahlundKanteAnswerEnum.TAIL, possibleOutcomesTail);

        final Map<KopfundZahlundKanteAnswerEnum, Double> possibleOutcomesHead = new LinkedHashMap<>();
        possibleOutcomesHead.put(
                KopfundZahlundKanteAnswerEnum.TAIL,
                (double) (Integer) playerData
                        .get(KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_TAIL));
        possibleOutcomesHead.put(
                KopfundZahlundKanteAnswerEnum.EDGE,
                (double) (Integer) playerData
                        .get(KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_EDGE));
        possibleOutcomesHead.put(
                KopfundZahlundKanteAnswerEnum.HEAD,
                (double) (Integer) playerData
                        .get(KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_HEAD_HEAD));
        possibleOutcomes.put(KopfundZahlundKanteAnswerEnum.HEAD, possibleOutcomesHead);

        final Map<KopfundZahlundKanteAnswerEnum, Double> possibleOutcomesEdge = new LinkedHashMap<>();
        possibleOutcomesEdge.put(
                KopfundZahlundKanteAnswerEnum.TAIL,
                (double) (Integer) playerData
                        .get(KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_EDGE_TAIL));
        possibleOutcomesEdge.put(
                KopfundZahlundKanteAnswerEnum.EDGE,
                (double) (Integer) playerData
                        .get(KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_EDGE_EDGE));
        possibleOutcomesEdge.put(
                KopfundZahlundKanteAnswerEnum.HEAD,
                (double) (Integer) playerData
                        .get(KopfundZahlundKanteGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_EDGE_HEAD));
        possibleOutcomes.put(KopfundZahlundKanteAnswerEnum.EDGE, possibleOutcomesEdge);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a KopfundZahl strategy.
     *
     * @param playerData The requested player data.
     * @return The KopfundZahl strategy.
     */
    private KopfundZahlundKanteStrategy getStrategy(final Map<String, Object> playerData) {
        return (KopfundZahlundKanteStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }

}
