package de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.AbstractPlayer;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKanteAnswerEnum;
import de.fhdw.gaming.ipspiel22.kopfundzahlundkante.domain.KopfundZahlundKantePlayer;

/**
 * Implements {@link KopfundZahlundKantePlayer}.
 */
public final class KopfundZahlundKantePlayerImpl extends AbstractPlayer<KopfundZahlundKantePlayer>
        implements KopfundZahlundKantePlayer {

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<KopfundZahlundKanteAnswerEnum, Map<KopfundZahlundKanteAnswerEnum, Double>> possibleOutcomes;
    /**
     * The answer of the player.
     */
    private Optional<KopfundZahlundKanteAnswerEnum> answer;

    /**
     * Creates a KopfundZahl player.
     *
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    public KopfundZahlundKantePlayerImpl(final String name,
            final Map<KopfundZahlundKanteAnswerEnum, Map<KopfundZahlundKanteAnswerEnum, Double>> possibleOutcomes) {
        super(name);
        this.possibleOutcomes = Collections.unmodifiableMap(
                new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
        this.answer = Optional.empty();
    }

    /**
     * Creates a KopfundZahl player.
     *
     * @param source The {@link KopfundZahlundKantePlayer} to copy.
     */
    KopfundZahlundKantePlayerImpl(final KopfundZahlundKantePlayer source) {
        super(source);
        this.possibleOutcomes = source.getPossibleOutcomes();
        this.answer = source.getAnswer();
    }

    @Override
    public String toString() {
        return String
                .format("KopfundZahlundKantePlayer[name=%s, state=%s, outcome=%s, answer=%s]", this.getName(),
                        this.getState(), this.getOutcome(), this.answer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof KopfundZahlundKantePlayerImpl) {
            final KopfundZahlundKantePlayerImpl other = (KopfundZahlundKantePlayerImpl) obj;
            return super.equals(obj) && this.answer.equals(other.answer);
        }
        return false;
    }

    @SuppressWarnings("PMD.UselessOverriding")
    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public Map<KopfundZahlundKanteAnswerEnum, Map<KopfundZahlundKanteAnswerEnum, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public Optional<KopfundZahlundKanteAnswerEnum> getAnswer() {
        return this.answer;
    }

    @Override
    public void setAnswer(final KopfundZahlundKanteAnswerEnum newAnswer) {
        if (this.answer.isPresent()) {
            throw new IllegalStateException(String.format("Player %s tried to change her answer.", this.getName()));
        }
        this.answer = Optional.of(newAnswer);
    }

    @Override
    public KopfundZahlundKantePlayer deepCopy() {
        return new KopfundZahlundKantePlayerImpl(this);
    }
}
