package de.schereSteinPapier.domain.impl;

import static de.schereSteinPapier.SSPConstants.AuswahlConstants.PAPIER;
import static de.schereSteinPapier.SSPConstants.AuswahlConstants.SCHERE;
import static de.schereSteinPapier.SSPConstants.AuswahlConstants.STEIN;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.schereSteinPapier.domain.SSPPlayer;
import de.schereSteinPapier.domain.SSPPlayerBuilder;

/**
 * Implements {@link SSPPlayerBuilder}.
 */
public final class SSPPlayerBuilderImpl implements SSPPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<String, Map<String, Double>>> possibleOutcomes;

    /**
     * Creates an {@link SSPPlayerBuilderImpl}.
     */
    SSPPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public SSPPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public SSPPlayerBuilder changePossibleOutcomes(final Map<String, Map<String, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public SSPPlayer build() throws GameException {
        return new SSPPlayerImpl(
                this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<String, Map<String, Double>> checkPossibleOutcomes(
            final Map<String, Map<String, Double>> outcomes) {
        this.checkPossibleOutcome(outcomes, SCHERE, SCHERE);
        this.checkPossibleOutcome(outcomes, SCHERE, STEIN);
        this.checkPossibleOutcome(outcomes, SCHERE, PAPIER);
        this.checkPossibleOutcome(outcomes, STEIN, STEIN);
        this.checkPossibleOutcome(outcomes, STEIN, SCHERE);
        this.checkPossibleOutcome(outcomes, STEIN, PAPIER);
        this.checkPossibleOutcome(outcomes, PAPIER, PAPIER);
        this.checkPossibleOutcome(outcomes, PAPIER, STEIN);
        this.checkPossibleOutcome(outcomes, PAPIER, SCHERE);
        return outcomes;
    }

    /**
     * Checks if a given outcome is defined for a player.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    private void checkPossibleOutcome(final Map<String, Map<String, Double>> outcomes, final String firstChoice,
            final String secondChoice) {
        if (outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                    String.format(
                            "No outcome defined for player '%s' and combination %s/%s.",
                            this.name,
                            firstChoice,
                            secondChoice));
        }
    }

//    /**
//     * Maps a boolean value to a "yes" or "no" answer.
//     *
//     * @param firstChoice The value to be mapped.
//     */
//    private static String toAnswer(final String firstChoice) {
//        return "yes";
//    }
}
