package de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDGameBuilder;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDGameBuilderFactory;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDPlayer;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDPlayerBuilder;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDStrategy;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.factory.GDDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.factory.GDStrategyFactory;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.factory.GDStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.moves.factory.GDMoveFactory;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.moves.impl.GDDefaultMoveFactory;

/**
 * Implements {@link GameBuilderFactory} by creating a Gefangenen Dilemma game builder.
 */
public class GDGameBuilderFactoryImpl implements GDGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Gefangenen Dilemma strategies.
     */
    private final Set<GDStrategy> strategies;

    /**
     * Creates a Gefangenen Dilemma game factory. Gefangenen Dilemma strategies are loaded by using the
     * {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public GDGameBuilderFactoryImpl() {
        this(new GDDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Gefangenen Dilemma game factory.
     *
     * @param strategyFactoryProvider The {@link GDStrategyFactoryProvider} for loading Gefangenen Dilemma strategies.
     */
    GDGameBuilderFactoryImpl(final GDStrategyFactoryProvider strategyFactoryProvider) {
        final GDMoveFactory moveFactory = new GDDefaultMoveFactory();

        final List<GDStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final GDStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Gefangenen Dilemma";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return GDGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return GDGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public GDGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final GDGameBuilder gameBuilder = new GDGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(GDGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(GDGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1");
            final GDPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final GDStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2");
            final GDPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final GDStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Gefangenen Dilemma game was aborted: %s", e.getMessage()),
                    e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        if (title.equals("Player 1")) {
            inputProvider
                    .needString(
                            GameBuilderFactory.PARAM_PLAYER_NAME,
                            "Name",
                            Optional.empty(),
                            new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                    .needInteger(
                            GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STATEM_STATEM,
                            "Player's outcome on Statement/Statement",
                            Optional.of(-8))
                    .needInteger(
                            GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STATEM_QUIET,
                            "Player's outcome on Statement/Quiet",
                            Optional.of(0))
                    .needInteger(
                            GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_QUIET_STATEM,
                            "Player's outcome on Quiet/Statement",
                            Optional.of(-10))
                    .needInteger(
                            GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_QUIET_QUIET,
                            "Player's outcome on Quiet/Quiet",
                            Optional.of(-1))
                    .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(),
                            this.strategies);
        } else {
            inputProvider
                    .needString(
                            GameBuilderFactory.PARAM_PLAYER_NAME,
                            "Name",
                            Optional.empty(),
                            new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                    .needInteger(
                            GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STATEM_STATEM,
                            "Player's outcome on Statement/Statement",
                            Optional.of(-8))
                    .needInteger(
                            GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STATEM_QUIET,
                            "Player's outcome on Statement/Quiet",
                            Optional.of(-10))
                    .needInteger(
                            GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_QUIET_STATEM,
                            "Player's outcome on Quiet/Statement",
                            Optional.of(0))
                    .needInteger(
                            GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_QUIET_QUIET,
                            "Player's outcome on Quiet/Quiet",
                            Optional.of(-1))
                    .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(),
                            this.strategies);
        }
        return inputProvider.requestData(title);
    }

    /**
     * Creates a Gefangenen Dilemma player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link GDPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private GDPlayer createPlayer(final GDPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<Boolean, Map<Boolean, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<Boolean, Double> possibleOutcomesStatement = new LinkedHashMap<>();
        possibleOutcomesStatement.put(
                Boolean.FALSE,
                (double) (Integer) playerData.get(GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STATEM_STATEM));
        possibleOutcomesStatement.put(
                Boolean.TRUE,
                (double) (Integer) playerData.get(GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_STATEM_QUIET));
        possibleOutcomes.put(Boolean.FALSE, possibleOutcomesStatement);

        final Map<Boolean, Double> possibleOutcomesQuiet = new LinkedHashMap<>();
        possibleOutcomesQuiet.put(
                Boolean.FALSE,
                (double) (Integer) playerData.get(GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_QUIET_STATEM));
        possibleOutcomesQuiet.put(
                Boolean.TRUE,
                (double) (Integer) playerData.get(GDGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_QUIET_QUIET));
        possibleOutcomes.put(Boolean.TRUE, possibleOutcomesQuiet);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a Gefangenen Dilemma strategy.
     *
     * @param playerData The requested player data.
     * @return The Gefangenen Dilemma strategy.
     */
    private GDStrategy getStrategy(final Map<String, Object> playerData) {
        return (GDStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
