/*
 * Copyright © 2020 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of VG-core.
 *
 * VG-core is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * VG-core is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VG-core. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel22.vierGewinnt.domain.impl;

import java.util.Objects;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGBoard;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGField;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGFieldState;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPosition;

/**
 * Implements {@link VGField}.
 */
final class VGFieldImpl implements VGField {

    /**
     * The board this field belongs to.
     */
    private final VGBoardImpl board;
    /**
     * The position at which this field is placed on the board.
     */
    private final VGPosition position;
    /**
     * The state of this field.
     */
    private VGFieldState state;

    /**
     * Creates an VG field.
     *
     * @param board    The board this field belongs to.
     * @param position The position at which this field is placed on the board.
     * @param state    The state of this field.
     */
    VGFieldImpl(final VGBoardImpl board, final VGPosition position, final VGFieldState state) {
        this.board = Objects.requireNonNull(board, "board");
        this.position = Objects.requireNonNull(position, "position");
        this.state = state;
    }

    @Override
    public String toString() {
        return String.format("VGField[position=%s, state=%s]", this.position, this.state);
    }

    /**
     * {@inheritDoc}
     * <p>
     * Does not compare the boards the fields belong to, respectively.
     */
    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof VGFieldImpl) {
            final VGFieldImpl other = (VGFieldImpl) obj;
            return this.position.equals(other.position) && this.state.equals(other.state);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.position, this.state);
    }

    @Override
    public VGBoard getBoard() {
        return this.board;
    }

    @Override
    public VGPosition getPosition() {
        return this.position;
    }

    @Override
    public VGFieldState getState() {
        return this.state;
    }

    /**
     * Sets the current state of this field. Does not change the states of neighbour fields.
     *
     * @param newState The new state.
     * @throws IllegalArgumentException if the new state is {@link VGFieldState#EMPTY} and different from the
     *                                  previous state.
     */
    @Override
    public void setState(final VGFieldState newState) throws IllegalArgumentException {
        final VGFieldState oldState = this.state;
        if (oldState.equals(newState)) {
            // nothing to do
            return;
        }

//        final boolean wasEmpty = oldState.equals(VGFieldState.EMPTY);
//        final boolean isEmpty = newState.equals(VGFieldState.EMPTY);
//
//        if (!wasEmpty && isEmpty) {
//            throw new IllegalArgumentException(
//                    String.format("The field at %s cannot become empty again.", this.position));
//        }

        this.state = newState;
        this.board.fieldChangedState(this, oldState);
    }

    @Override
    public void placeToken(final boolean redChip) throws GameException {
        final VGFieldState newState = redChip ? VGFieldState.RED : VGFieldState.YELLOW;
        this.setState(newState);
    }
}
