package de.fhdw.gaming.ipspiel22.kopfundzahl.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.AbstractPlayer;
import de.fhdw.gaming.ipspiel22.kopfundzahl.domain.KopfundZahlPlayer;

/**
 * Implements {@link KopfundZahlPlayer}.
 */
public final class KopfundZahlPlayerImpl extends AbstractPlayer<KopfundZahlPlayer> implements KopfundZahlPlayer {

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<Boolean, Map<Boolean, Double>> possibleOutcomes;
    /**
     * The answer of the player.
     */
    private Optional<Boolean> answer;

    /**
     * Creates a KopfundZahl player.
     *
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    public KopfundZahlPlayerImpl(final String name, final Map<Boolean, Map<Boolean, Double>> possibleOutcomes) {
        super(name);
        this.possibleOutcomes = Collections.unmodifiableMap(
                new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
        this.answer = Optional.empty();
    }

    /**
     * Creates a KopfundZahl player.
     *
     * @param source The {@link KopfundZahlPlayer} to copy.
     */
    KopfundZahlPlayerImpl(final KopfundZahlPlayer source) {
        super(source);
        this.possibleOutcomes = source.getPossibleOutcomes();
        this.answer = source.getAnswer();
    }

    @Override
    public String toString() {
        return String
                .format("KopfundZahlPlayer[name=%s, state=%s, outcome=%s, answer=%s]", this.getName(),
                        this.getState(), this.getOutcome(), this.answer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof KopfundZahlPlayerImpl) {
            final KopfundZahlPlayerImpl other = (KopfundZahlPlayerImpl) obj;
            return super.equals(obj) && this.answer.equals(other.answer);
        }
        return false;
    }

    @SuppressWarnings("PMD.UselessOverriding")
    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public Map<Boolean, Map<Boolean, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public Optional<Boolean> getAnswer() {
        return this.answer;
    }

    @Override
    public void setAnswer(final boolean newAnswer) {
        if (this.answer.isPresent()) {
            throw new IllegalStateException(String.format("Player %s tried to change her answer.", this.getName()));
        }
        this.answer = Optional.of(newAnswer);
    }

    @Override
    public KopfundZahlPlayer deepCopy() {
        return new KopfundZahlPlayerImpl(this);
    }
}
