package de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.AbstractPlayer;
import de.fhdw.gaming.ipspiel22.gefangenenDilemma.domain.GDPlayer;

/**
 * Implements {@link GDPlayer}.
 */
public class GDPlayerImpl extends AbstractPlayer<GDPlayer> implements GDPlayer {

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<Boolean, Map<Boolean, Double>> possibleOutcomes;
    /**
     * The answer of the player.
     */
    private Optional<Boolean> answer;

    /**
     * Creates a Gefangenen Dilemma player.
     *
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    GDPlayerImpl(final String name, final Map<Boolean, Map<Boolean, Double>> possibleOutcomes) {
        super(name);
        this.possibleOutcomes = Collections
                .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
        this.answer = Optional.empty();
    }

    /**
     * Creates a Gefangenen Dilemma player.
     *
     * @param source The {@link GDPlayer} to copy.
     */
    GDPlayerImpl(final GDPlayer source) {
        super(source);
        this.possibleOutcomes = source.getPossibleOutcomes();
        this.answer = source.getAnswer();
    }

    @Override
    public String toString() {
        return String
                .format("GDPlayer[name=%s, state=%s, outcome=%s, answer=%s]", this.getName(), this.getState(),
                        this.getOutcome(), this.answer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof GDPlayerImpl) {
            final GDPlayerImpl other = (GDPlayerImpl) obj;
            return super.equals(obj) && this.answer.equals(other.answer);
        }
        return false;
    }

    @SuppressWarnings("PMD.UselessOverriding")
    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public Map<Boolean, Map<Boolean, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public Optional<Boolean> getAnswer() {
        return this.answer;
    }

    @Override
    public void setAnswer(final boolean newAnswer) {
        if (this.answer.isPresent()) {
            throw new IllegalStateException(String.format("Player %s tried to change her answer.", this.getName()));
        }
        this.answer = Optional.of(newAnswer);
    }

    @Override
    public GDPlayer deepCopy() {
        return new GDPlayerImpl(this);
    }
}
