package de.fhdw.gaming.ipspiel22.vierGewinnt.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGGameBuilder;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGGameBuilderFactory;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPlayer;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGPlayerBuilder;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.VGStrategy;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.factory.VGDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.factory.VGStrategyFactory;
import de.fhdw.gaming.ipspiel22.vierGewinnt.domain.factory.VGStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel22.vierGewinnt.moves.factory.VGMoveFactory;
import de.fhdw.gaming.ipspiel22.vierGewinnt.moves.impl.VGDefaultMoveFactory;

/**
 * Implements {@link GameBuilderFactory} by creating a Vier gewinnt game builder.
 */
public class VGGameBuilderFactoryImpl implements VGGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Vier gewinnt strategies.
     */
    private final Set<VGStrategy> strategies;

    /**
     * Creates a Vier gewinnt game factory. Vier gewinnt strategies are loaded by using the
     * {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public VGGameBuilderFactoryImpl() {
        this(new VGDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Vier gewinnt game factory.
     *
     * @param strategyFactoryProvider The {@link VGStrategyFactoryProvider} for loading Vier gewinnt strategies.
     */
    VGGameBuilderFactoryImpl(final VGStrategyFactoryProvider strategyFactoryProvider) {
        final VGMoveFactory moveFactory = new VGDefaultMoveFactory();

        final List<VGStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final VGStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Vier gewinnt";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return VGGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return VGGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public VGGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final VGGameBuilder gameBuilder = new VGGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(VGGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(VGGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1",
                    Optional.empty());
            final VGPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final VGStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2",
                    Optional.of(!(Boolean) firstPlayerData
                            .get(VGGameBuilderFactory.PARAM_PLAYER_USING_RED_CHIPS)));
            final VGPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final VGStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Vier gewinnt game was aborted: %s", e.getMessage()),
                    e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @param usingRedChips If set, determines the mandatory colour of the player. Otherwise, the colour is initially
     *                      set to red, and the user can change it at will.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title,
            final Optional<Boolean> usingRedChips)
            throws GameException, InputProviderException {

        inputProvider
                .needString(
                        GameBuilderFactory.PARAM_PLAYER_NAME,
                        "Name",
                        Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needBoolean(
                        VGGameBuilderFactory.PARAM_PLAYER_USING_RED_CHIPS,
                        "Red colour",
                        Optional.of(Boolean.TRUE))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        if (usingRedChips.isPresent()) {
            inputProvider.fixedBoolean(
                    VGGameBuilderFactory.PARAM_PLAYER_USING_RED_CHIPS,
                    usingRedChips.get());
        }

        return inputProvider.requestData(title);
    }

    /**
     * Creates a Vier gewinnt player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return {@link VGPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private VGPlayer createPlayer(final VGPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {
        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changeUsingRedChips((Boolean) playerData.get(VGGameBuilderFactory.PARAM_PLAYER_USING_RED_CHIPS))
                .build();
    }

    /**
     * Returns a Vier gewinnt strategy.
     *
     * @param playerData The requested player data.
     * @return The Vier gewinnt strategy.
     */
    private VGStrategy getStrategy(final Map<String, Object> playerData) {
        return (VGStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
