package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4GameBuilder;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4GameBuilderFactory;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4PlayerBuilder;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.validation.C4BoardLimits;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.validation.C4BoardValidator;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.validation.C4BoardValidatorInfo;
import de.fhdw.gaming.ipspiel23.c4.moves.factory.IC4MoveFactory;
import de.fhdw.gaming.ipspiel23.c4.moves.impl.C4DefaultMoveFactory;
import de.fhdw.gaming.ipspiel23.c4.strategies.C4DefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4Strategy;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4StrategyFactory;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4StrategyFactoryProvider;
import de.fhdw.gaming.ipspiel23.c4.utils.ByRef;
import de.fhdw.gaming.ipspiel23.c4.utils.MakeRef;

import static de.fhdw.gaming.ipspiel23.c4.domain.impl.validation.C4BoardLimits.MAX_NUMBER_OF_PLAYERS;
import static de.fhdw.gaming.ipspiel23.c4.domain.impl.validation.C4BoardLimits.MIN_NUMBER_OF_PLAYERS;

/**
 * The default implementation of {@link IC4GameBuilderFactory}.
 */
public final class C4GameBuilderFactory implements IC4GameBuilderFactory {

    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * The set of strategies that are available for the game.
     */
    private final Set<IC4Strategy> strategies;
    
    /**
     * Creates a new instance of {@link C4GameBuilderFactory} with the default strategy factory provider.
     */
    public C4GameBuilderFactory() {
        this(new C4DefaultStrategyFactoryProvider());
    }

    /**
     * Creates a new instance of {@link C4GameBuilderFactory} with the given strategy factory provider.
     * 
     * @param strategyFactoryProvider The strategy factory provider to use.
     */
    public C4GameBuilderFactory(final IC4StrategyFactoryProvider strategyFactoryProvider) {
        final IC4MoveFactory moveFactory = new C4DefaultMoveFactory();
        final List<IC4StrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new HashSet<>();
        for (final IC4StrategyFactory factory : factories) {
            final IC4Strategy strategy = factory.create(moveFactory);
            strategies.add(strategy);
        }
    }

    @Override
    public String getName() {
        // int.MAX / 2 (because 2 is the minimum solution size)
        // at least in theory every player should be allowed to make 
        // solution size moves :)
        return "Connect 2 <= N <= 1,073,741,823";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return MIN_NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return MAX_NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @SuppressWarnings("unchecked")
    @Override
    public IC4GameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final IC4GameBuilder gameBuilder = new C4GameBuilder();
            
            // basic game properties
            final Map<String, Object> dataSet = inputProvider
                .needInteger(IC4GameBuilderFactory.PARAM_PLAYER_COUNT,
                    "Number of players",
                    Optional.of(IC4GameBuilder.DEFAULT_PLAYER_COUNT),
                    new MinValueValidator<>(MIN_NUMBER_OF_PLAYERS),
                    new MaxValueValidator<>(MAX_NUMBER_OF_PLAYERS))
                .needInteger(IC4GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                    "Maximum computation time per move in seconds",
                    Optional.of(IC4GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MinValueValidator<>(MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MaxValueValidator<>(MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                .requestData("Game properties");
            
            final int maxComputationTime = (int) dataSet.get(IC4GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE);
            gameBuilder.changeMaximumComputationTimePerMove(maxComputationTime);
            
            final int playerCount = (int) dataSet.get(IC4GameBuilderFactory.PARAM_PLAYER_COUNT);

            // players
            // we need to keep track of the input provider and data set as they have
            // to be passed to the next player input providers
            // therefore "pass by reference" or well... just wrap them in a reference type
            // to emulate that behavior :D 
            // In C you'd just pass the addresses of inputProvider and dataSet
            final ByRef<InputProvider> inputProviderRef = MakeRef.of(inputProvider);
            final ByRef<Map<String, Object>> dataSetRef = MakeRef.of(dataSet);
            for (int i = 0; i < playerCount; i++) {
                addPlayer(inputProviderRef, dataSetRef, gameBuilder);
            }

            // board properties
            // we have cross dependencies relevant for validation between the board properties.
            // therefore we share a validation state between the responsible input validators
            final C4BoardLimits sharedValidationState = new C4BoardLimits(playerCount, 
                IC4GameBuilder.DEFAULT_BOARD_ROWS, 
                IC4GameBuilder.DEFAULT_BOARD_COLUMNS, 
                IC4GameBuilder.DEFAULT_REQUIRED_SOLUTION_SIZE);
             
            final C4BoardValidator rowCountValidator = new C4BoardValidator(sharedValidationState, 
                context -> context.getState().setRowCount(context.getValue()), C4BoardValidatorInfo.ROW_COUNT);
            final C4BoardValidator columnCountValidator = new C4BoardValidator(sharedValidationState, 
                context -> context.getState().setColumnCount(context.getValue()), C4BoardValidatorInfo.COLUMN_COUNT);
            final C4BoardValidator solutionSizeValidator = new C4BoardValidator(sharedValidationState, 
                context -> context.getState().setSolutionSize(context.getValue()), C4BoardValidatorInfo.SOLUTION_SIZE);
            
            final InputProvider boardInputProvider = inputProviderRef.getValue().getNext(dataSetRef.getValue());
            final Map<String, Object> boardDataSet = boardInputProvider
                .needInteger(IC4GameBuilderFactory.PARAM_BOARD_ROWS, 
                    "Number of rows",
                    Optional.of(IC4GameBuilder.DEFAULT_BOARD_ROWS),
                    rowCountValidator)
                .needInteger(IC4GameBuilderFactory.PARAM_BOARD_COLUMNS, 
                    "Number of columns",
                    Optional.of(IC4GameBuilder.DEFAULT_BOARD_COLUMNS),
                    columnCountValidator)
                .needInteger(IC4GameBuilderFactory.PARAM_REQUIRED_SOLUTION_SIZE, 
                    "Number of consecutive pieces required to win",
                    Optional.of(IC4GameBuilder.DEFAULT_REQUIRED_SOLUTION_SIZE),
                    solutionSizeValidator)
                .requestData("Board properties");

            // assert the provided state is valid
            sharedValidationState.assertIsValid();
            
            // apply changes
            gameBuilder
                .changeBoardRows((int) boardDataSet.get(IC4GameBuilderFactory.PARAM_BOARD_ROWS))
                .changeBoardColumns((int) boardDataSet.get(IC4GameBuilderFactory.PARAM_BOARD_COLUMNS))
                .changeRequiredSolutionSize((int) boardDataSet.get(IC4GameBuilderFactory.PARAM_REQUIRED_SOLUTION_SIZE));

            // done! :)
            return gameBuilder;

        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating the Connect N game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Adds a player to the game.
     * 
     * @param inputProvider A reference to the input provider to use (InputProvider pointer)
     * @param lastDataSet A reference to the last data set (Map<String, Object> pointer)
     * @param gameBuilder The game builder to use.
     * @throws InputProviderException If the input provider fails.
     * @throws GameException If the game builder fails.
     */
    @SuppressWarnings("unchecked")
    private void addPlayer(final ByRef<InputProvider> inputProvider, 
            final ByRef<Map<String, Object>> lastDataSet,
            final IC4GameBuilder gameBuilder) throws InputProviderException, GameException {
        
        final IC4PlayerBuilder playerBuilder = gameBuilder.createPlayerBuilder();
        final int playerToken = playerBuilder.getToken();
        final String title = String.format("Player %d", playerToken);

        final InputProvider playerInputProvider = inputProvider.getValue().getNext(lastDataSet.getValue());

        final Map<String, Object> dataSet = playerInputProvider
            .needString(IC4GameBuilderFactory.PARAM_PLAYER_NAME,
                "Name", 
                Optional.empty(),
                new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
            .needObject(IC4GameBuilderFactory.PARAM_PLAYER_STRATEGY, 
                "Strategy", 
                Optional.empty(), 
                this.strategies)
            .requestData(title);
        
        final String name = (String) dataSet.get(IC4GameBuilderFactory.PARAM_PLAYER_NAME);
        final IC4Strategy strategy = (IC4Strategy) dataSet.get(IC4GameBuilderFactory.PARAM_PLAYER_STRATEGY);

        final IC4Player player = playerBuilder
            .changeName(name)
            .build();

        gameBuilder.addPlayer(player, strategy);

        // update ref parameters
        inputProvider.setValue(playerInputProvider);
        lastDataSet.setValue(dataSet);
    }
}