package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.Optional;
import java.util.function.Consumer;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4PlayerBuilder;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4Strategy;

/**
 * The default implementation of {@link IC4PlayerBuilder}.
 */
final class C4PlayerBuilder implements IC4PlayerBuilder {

    /**
     * The token that uniquely identifies the player.
     */
    private final int token;
    
    /**
     * The name of the player.
     */
    private Optional<String> name;

    /**
     * The hook that is used to inject the strategy into the player object when the player is added to the game.
     */
    private Consumer<IC4Strategy> playerStrategyHook; 

    /**
     * Creates a new instance of {@link C4PlayerBuilder}.
     * 
     * @param id The token that uniquely identifies the player.
     */
    C4PlayerBuilder(final int id) {
        this.token = id;
        this.name = Optional.empty();
    }

    /**
     * Sets the hook that is used to inject the strategy into the player object when the player is added to the game.
     * 
     * @param value The hook that is used to inject the strategy into the player object when the player is 
     * added to the game.
     */
    void setPlayerStrategyHook(final Consumer<IC4Strategy> value) {
        this.playerStrategyHook = value;
    }

    /**
     * Injects the strategy into the player object.
     * 
     * @param strategy The strategy that is injected into the player object.
     */
    void injectPlayerStrategyUsingHook(final IC4Strategy strategy) {
        if (playerStrategyHook == null) {
            throw new IllegalStateException("Attempted to inject strategy into player object before "
                + "hook was configured!");
        }
        // we don't want to expose the strategy setter to anyone, so it's done via this
        // hook kindly provided to us by the C4Player constructor.
        playerStrategyHook.accept(strategy);
    }

    @Override
    public IC4PlayerBuilder changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public IC4Player build() throws GameException {
        return new C4Player(this, this.token, this.name.orElseThrow());
    }

    @Override
    public int getToken() {
        return this.token;
    }
}
