/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel23-demo.
 *
 * Ipspiel23-demo is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel23-demo is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel23-demo. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel23.demo.domain.impl;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.demo.domain.DemoPlayer;
import de.fhdw.gaming.ipspiel23.demo.domain.DemoPlayerBuilder;

/**
 * Implements {@link DemoPlayerBuilder}.
 */
final class DemoPlayerBuilderImpl implements DemoPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<Boolean, Map<Boolean, Double>>> possibleOutcomes;

    /**
     * Creates an {@link DemoPlayerBuilderImpl}.
     */
    DemoPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public DemoPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public DemoPlayerBuilder changePossibleOutcomes(final Map<Boolean, Map<Boolean, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public DemoPlayer build() throws GameException {
        return new DemoPlayerImpl(
            this.name.orElseThrow(),
            this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<Boolean, Map<Boolean, Double>> checkPossibleOutcomes(
            final Map<Boolean, Map<Boolean, Double>> outcomes) {
        this.checkPossibleOutcome(outcomes, false, false);
        this.checkPossibleOutcome(outcomes, false, true);
        this.checkPossibleOutcome(outcomes, true, false);
        this.checkPossibleOutcome(outcomes, true, true);
        return outcomes;
    }

    /**
     * Checks if a given outcome is defined for a player.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    private void checkPossibleOutcome(final Map<Boolean, Map<Boolean, Double>> outcomes, final boolean firstChoice,
            final boolean secondChoice) {
        if (outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                String.format(
                    "No outcome defined for player '%s' and combination %s/%s.",
                    this.name,
                    toAnswer(firstChoice),
                    toAnswer(secondChoice)));
        }
    }

    /**
     * Maps a boolean value to a "yes" or "no" answer.
     *
     * @param value The value to be mapped.
     */
    private static String toAnswer(final boolean value) {
        return value ? "yes" : "no";
    }
}
