package de.fhdw.gaming.ipspiel23.dilemma.domain.internals;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import de.fhdw.gaming.core.domain.DefaultObserverFactoryProvider;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaGame;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaGameBuilder;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayerBuilder;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaStrategy;
import de.fhdw.gaming.ipspiel23.dilemma.moves.internals.DilemmaMove;

/**
 * Implements {@link IDilemmaGameBuilder}.
 */
public class DilemmaGameBuilder implements IDilemmaGameBuilder {
    
    /**
     * The {@link ObserverFactoryProvider}.
     */
    private ObserverFactoryProvider observerFactoryProvider;

    /**
     * The players participating in this game.
     */
    private final List<IDilemmaPlayer> players;

     /**
     * The player builders used to create the players. 
     * (player name -> builder)
     */
    private final Map<String, DilemmaPlayerBuilder> playerBuilders;

    /**
     * The maximum computation time per move in seconds.
     */
    private int maxComputationTimePerMove;

    /**
     * Creates a Dilemma game builder.
     */
    public DilemmaGameBuilder() {
        this.players = new ArrayList<>(2);
        this.playerBuilders = new HashMap<>();
        this.observerFactoryProvider = new DefaultObserverFactoryProvider();
        this.maxComputationTimePerMove = GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE;
    }

    @Override
    public IDilemmaPlayerBuilder createPlayerBuilder() {
        return new DilemmaPlayerBuilder(this);
    }

    /**
     * Called by {@link DilemmaPlayerBuilder} when a player is built. 
     * Required for strategy injection.
     * @param builder The builder that built the player.
     * @param player The player that was built.
     */
    void onPlayerBuilt(final DilemmaPlayerBuilder builder, final IDilemmaPlayer player) {
        this.playerBuilders.put(player.getName(), builder);
    }

    @Override
    public IDilemmaGameBuilder addPlayer(final IDilemmaPlayer player, final IDilemmaStrategy strategy)
            throws GameException {
        
        if (this.players.size() >= 2) {
            throw new GameException(String.format("More than two players are not allowed."));
        }
        final DilemmaPlayerBuilder builder = playerBuilders.getOrDefault(player.getName(), null);
        if (builder == null) {
            throw new GameException("Attempted to add player that was not built using a " 
                + DilemmaPlayerBuilder.class.getSimpleName());
        }
        // we need a way to store the strategy in the player instance (because that's where it belongs)
        // that is handled via an internal callback set in the ctor of the player.
        // We don't want client code to be able to modify strategies, so no simple public setters :)
        builder.injectPlayerStrategyUsingHook(player, strategy);
        players.add(player);
        return this;
    }

    @Override
    public IDilemmaGameBuilder changeMaximumComputationTimePerMove(final int newMaxComputationTimePerMove) {
        this.maxComputationTimePerMove = newMaxComputationTimePerMove;
        return this;
    }

    @Override
    public IDilemmaGameBuilder changeObserverFactoryProvider(final ObserverFactoryProvider newObserverFactoryProvider) {
        this.observerFactoryProvider = newObserverFactoryProvider;
        return this;
    }

    /**
     * Builds the game with the predefined parameters.
     */
    @Override
    public IDilemmaGame build(final int id) throws GameException, InterruptedException {
        if (this.players.size() != 2) {
            throw new GameException("A Dilemma game needs exactly two players.");
        }

        final DilemmaState initialState = new DilemmaState(this.players.get(0), this.players.get(1));
        final Map<String, IDilemmaStrategy> strategies = this.players
            .stream()
            .collect(Collectors.toMap(p -> p.getName(), p -> p.getStrategy()));

        return new DilemmaGame(
            id,
            initialState,
            strategies,
            this.maxComputationTimePerMove,
            DilemmaMove.class::isInstance,
            this.observerFactoryProvider);
    }
}
