package de.fhdw.gaming.ipspiel23.dilemma.strategy.internals;

import de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMove;

/**
 * The record of a single round of a dilemma game.
 * @param player1Data The round data of the first player.
 * @param player2Data The round data of the second player.
 */
public record DilemmaRoundData(DilemmaRoundPlayerData player1Data, DilemmaRoundPlayerData player2Data) {

    /**
     * Returns the round data for the given player.
     * @param player The player to get the data for. 
     * If no player in the round data set matches the supplied player 
     * object, an {@link IllegalArgumentException} will be thrown.
     * @return The round data for the given player.
     */
    public DilemmaRoundPlayerData forPlayer(final IDilemmaPlayer player) {
        if (player1Data.player().getName().equals(player.getName())) {
            return player1Data;
        }
        if (player2Data.player().getName().equals(player.getName())) {
            return player2Data;
        }
        throw new IllegalArgumentException("Unknown player: " + player);
    }

    /**
     * Returns the round data for the opponent of the given player.
     * @param player The player to get the data of the opponent for.
     * If no player in the round data set matches the supplied player 
     * object, an {@link IllegalArgumentException} will be thrown.
     * @return The round data for the opponent of the given player.
     */
    public DilemmaRoundPlayerData forOpponentOf(final IDilemmaPlayer player) {
        if (player1Data.player().getName().equals(player.getName())) {
            return player2Data;
        }
        if (player2Data.player().getName().equals(player.getName())) {
            return player1Data;
        }
        throw new IllegalArgumentException("Unknown player: " + player);
    }

    /**
     * Creates a new round data record from the given raw data.
     * @param player1 The first player.
     * @param player1Move The move of the first player.
     * @param player2 The second player.
     * @param player2Move The move of the second player.
     * @return The new round data record.
     */
    public static DilemmaRoundData fromRaw(final IDilemmaPlayer player1, final IDilemmaMove player1Move, 
            final IDilemmaPlayer player2, final IDilemmaMove player2Move) {
        final DilemmaAnswerType player1Answer = 
            player1Move == null
                ? null
                : player1Move.getAnswer();
        final DilemmaAnswerType player2Answer = 
            player2Move == null
                ? null
                : player2Move.getAnswer();
        final var player1Data = new DilemmaRoundPlayerData(player1, player1Answer, player1Move);
        final var player2Data = new DilemmaRoundPlayerData(player2, player2Answer, player2Move);
        return new DilemmaRoundData(player1Data, player2Data);
    }
}
