package de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.imperfect_tit_for_tat;

import java.util.Optional;
import java.util.Random;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaState;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMove;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMoveFactory;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.DilemmaRoundData;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.internals.tit_for_tat.DilemmaTitForTatStrategy;
import de.fhdw.gaming.ipspiel23.memory.IGameMemory;

/**
 * Tit-For-Tat, but imitates opponent's last move with high (but less than one) probability.
 */
public class DilemmaImperfectTitForTatStrategy extends DilemmaTitForTatStrategy {

    /**
     * The probability with which to play TFT.
     */
    static final double IMITATION_PROBABILITY = 0.9d;

    /**
     * RNG.
     */
    private static final Random RANDOM = new Random();

    /**
     * Creates a new instance of the {@link DilemmaImperfectTitForTatStrategy} class.
     * @param moveFactory The move factory to use.
     */
    DilemmaImperfectTitForTatStrategy(final IDilemmaMoveFactory moveFactory) {
        super(moveFactory);
    }

    @Override
    public Optional<IDilemmaMove> computeNextMove(final int gameId, final IDilemmaPlayer player, 
            final IDilemmaState state)
            throws GameException, InterruptedException {
        // if this is the first move, always cooperate (no previous move to imitate with less than 100% probability)
        final IGameMemory<DilemmaRoundData> memory = getMemoryForPlayer(player, state);
        final IDilemmaMoveFactory moveFactory = getMoveFactory();
        if (memory.size() == 0) {
            return Optional.of(moveFactory.createCooperateMove());
        }
        // be lazy about it and see what TFT would normally return ...
        final IDilemmaMove perfectTftMove = super.computeNextMove(gameId, player, state).get();
        // ... now see what RNGsus has to say about that:
        return Optional.of(RANDOM.nextDouble() > IMITATION_PROBABILITY
            ? moveFactory.oppositeOf(perfectTftMove)
            : perfectTftMove);
    }

    @Override 
    public String toString() {
        return String.format("%s[imitationProbability: %f]", getClass().getSimpleName(), IMITATION_PROBABILITY);
    }
}
