package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.Optional;

import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4Strategy;

/**
 * The default implementation of {@link IC4Player}.
 */
public class C4Player implements IC4Player {

    /**
     * The name of the player.
     */
    private final String name;

    /**
     * The token uniquely identifying the player.
     */
    private final int token;
    
    /**
     * The state of the player.
     * @hidden this is an attempt to get PMD to shut up about the "DataClass" warning
     */
    private final C4MutablePlayerState mutableState;

    /**
     * The strategy of the player.
     */
    private IC4Strategy strategy;
    
    /**
     * Creates a new instance.
     * 
     * @param builder The parent player builder that will be receive the strategy hook for the player.
     * @param token The token uniquely identifying the player.
     * @param name The name of the player.
     */
    C4Player(final C4PlayerBuilder builder, final int token, final String name) {
        this.token = token;
        this.name = name;
        builder.setPlayerStrategyHook(newStrategy -> this.strategy = newStrategy);

        this.mutableState = new C4MutablePlayerState(PlayerState.PLAYING, null);
    }
    
    /**
     * Copy constructor.
     * 
     * @param player The player to copy.
     */
    private C4Player(final C4Player player) {
        this.token = player.token;
        this.name = player.name;
        this.mutableState = new C4MutablePlayerState(player.mutableState.getState(), 
            player.mutableState.getOutcome().orElse(null));
        this.strategy = player.strategy;
    }
    
    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public PlayerState getState() {
        return this.mutableState.getState();
    }

    @Override
    public void setState(final PlayerState newState) {
        this.mutableState.setState(newState);
    }

    @Override
    public Optional<Double> getOutcome() {
        return this.mutableState.getOutcome();
    }

    @Override
    public void setOutcome(final double newOutcome) {
        this.mutableState.setOutcome(newOutcome);
    }

    @Override
    public IC4Player deepCopy() {
        return new C4Player(this);
    }

    @Override
    public int getToken() {
        return this.token;
    }

    @Override
    public IC4Strategy getStrategy() {
        if (this.strategy == null) {
            throw new IllegalStateException("The strategy of the player " + this.name + " was never set.");
        }
        return this.strategy;
    }

    @Override
    public boolean equals(final Object other) {
        if (other == null) {
            return false;
        }
        if (!(other instanceof IC4Player)) {
            return false;
        }
        final IC4Player otherPlayer = (IC4Player) other;
        return this.token == otherPlayer.getToken() 
            && this.name.equals(otherPlayer.getName()) 
            && this.getState() == otherPlayer.getState()
            && this.getOutcome().equals(otherPlayer.getOutcome())
            && this.getStrategy().equals(otherPlayer.getStrategy());
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 31 * hash + this.token;
        hash = 31 * hash + this.name.hashCode();
        hash = 31 * hash + this.mutableState.hashCode();
        if (this.strategy != null) {
            hash = 31 * hash + this.strategy.hashCode();
        }
        return hash;
    }

    @Override
    public String toString() {
        return "C4Player [name=" + this.name 
            + ", token=" + this.token 
            + ", state=" + this.getState() 
            + ", outcome=" + this.getOutcome() 
            + ", strategy=" + this.strategy + "]";
    }
}
