package de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgGameBuilder;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgGameBuilderFactory;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgPlayer;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgPlayerBuilder;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgStrategy;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.factory.FzgStrategyDefaultFactoryProvider;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.factory.FzgStrategyFactory;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.factory.FzgStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.move.AnswerOptions;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.move.factory.AnswerFactory;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.move.impl.AnswerFactoryImpl;

/**
 * Creates game builder, implements {@link FzgGameBuilderFactory}.
 * 
 */
public class FzgGameBuilderFactoryImpl implements FzgGameBuilderFactory {

    /**
     * Name of the game.
     */
    private static final String GAME_NAME = "Freizeitgestaltung";
    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Demo strategies.
     */
    private final Set<FzgStrategy> strategies;

    /**
     * Creates gameBuilder Factory.
     */
    public FzgGameBuilderFactoryImpl() {
        this(new FzgStrategyDefaultFactoryProvider());
    }

    /**
     * Creates a Demo game factory.
     *
     * @param strategyFactoryProvider The {@link DemoStrategyFactoryProvider} for loading Demo strategies.
     */
    FzgGameBuilderFactoryImpl(final FzgStrategyFactoryProvider strategyFactoryProvider) {
        final AnswerFactory answerFactory = new AnswerFactoryImpl();

        final List<FzgStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final FzgStrategyFactory factory : factories) {
            this.strategies.add(factory.create(answerFactory));
        }
    }

    @Override
    public String getName() {
        return GAME_NAME;
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public FzgGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final FzgGameBuilder gameBuilder = new FzgGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(FzgGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(FzgGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String,
                    Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1 / Gabi", true);
            final FzgPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final FzgStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String,
                    Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2 / Klaus",
                            false);
            final FzgPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final FzgStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Demo game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * returns data for PlayerBuilder.
     * 
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @param isGabi        Flag to show if the Player is "Gabi"/FirstPlayer, therefore different standard-outcome.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title,
            final Boolean isGabi)
            throws GameException, InputProviderException {
        inputProvider
                .needString(
                        GameBuilderFactory.PARAM_PLAYER_NAME,
                        "Name",
                        Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(
                        FzgGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CIN_CIN,
                        "Player's outcome on Cinema/Cinema",
                        isGabi ? Optional.of(2) : Optional.of(1))
                .needInteger(
                        FzgGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CIN_FOOT,
                        "Player's outcome on Cinema/Football",
                        Optional.of(0))
                .needInteger(
                        FzgGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_FOOT_CIN,
                        "Player's outcome on Football/Cinema",
                        Optional.of(0))
                .needInteger(
                        FzgGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_FOOT_FOOT,
                        "Player's outcome on Football/Football",
                        isGabi ? Optional.of(1) : Optional.of(2))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Creates FzgPlayer.
     * 
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link FzgPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private FzgPlayer createPlayer(final FzgPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<AnswerOptions, Map<AnswerOptions, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<AnswerOptions, Double> possibleOutcomesCinema = new LinkedHashMap<>();
        possibleOutcomesCinema.put(
                AnswerOptions.CINEMA,
                (double) (Integer) playerData.get(FzgGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CIN_CIN));
        possibleOutcomesCinema.put(
                AnswerOptions.FOOTBALL,
                (double) (Integer) playerData.get(FzgGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CIN_FOOT));
        possibleOutcomes.put(AnswerOptions.CINEMA, possibleOutcomesCinema);

        final Map<AnswerOptions, Double> possibleOutcomesFootball = new LinkedHashMap<>();
        possibleOutcomesFootball.put(
                AnswerOptions.CINEMA,
                (double) (Integer) playerData.get(FzgGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_FOOT_CIN));
        possibleOutcomesFootball.put(
                AnswerOptions.FOOTBALL,
                (double) (Integer) playerData.get(FzgGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_FOOT_FOOT));
        possibleOutcomes.put(AnswerOptions.FOOTBALL, possibleOutcomesFootball);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a FzgStrategy.
     *
     * @param playerData The requested player data.
     * @return The  FzgStrategy.
     */
    private FzgStrategy getStrategy(final Map<String, Object> playerData) {
        return (FzgStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
