package de.fhdw.gaming.ipspiel23.c4.domain;

import java.util.Optional;
import java.util.Set;

/**
 * Represents an external facade for object oriented access to the connect four board.
 */
public interface IC4Board extends IC4BoardBase {

    /**
     * Retrieves the internal board for usage in performance critical scenarios.
     */
    IC4BoardSlim getInternalBoard();

    /**
     * Retrieves the field at the specified position.
     * @param position The position of the field.
     * @return The field at the specified position, if it exists.
     */
    Optional<IC4Field> tryGetField(IC4Position position);

    /**
     * Retrieves the field at the specified position.
     * @param position The position of the field.
     * @return The field at the specified position.
     * @throws IndexOutOfBoundsException If the specified position is outside the bounds of the board.
     */
    IC4Field getField(IC4Position position);

    /**
     * Retrieves all fields of the board in their object representation, not very efficient.
     * @return All fields of the board.
     */
    IC4Field[][] getFields();

    /**
     * Lazy evaluation for the first solution, e.g. the first n-in-a-row (where n is the number of 
     * tokens in a row required to win)
     * @return the first solution, or {@link Optional#empty()} if no solution exists for the current board state
     */
    Optional<IC4Solution> tryFindFirstSolution();

    /**
     * Eager evaluation for all solutions, e.g. all n-in-a-row (where n is the number of 
     * tokens in a row required to win)
     * @return all solutions, or an empty set if no solution exists for the current board state
     */
    Set<IC4Solution> findAllSolutions();

    /**
     * Checks whether the specified position is empty.
     * @param position The position to check.
     * @return {@code true} if the specified position is empty, {@code false} otherwise.
     * @throws IndexOutOfBoundsException If the specified position is outside the bounds of the board.
     */
    boolean isEmpty(IC4Position position);

    /**
     * Checks whether the specified position is empty.
     * @param row The row of the position to check.
     * @param column The column of the position to check.
     * @return {@code true} if the specified position is empty, {@code false} otherwise.
     * @throws IndexOutOfBoundsException If the specified position is outside the bounds of the board.
     */
    boolean isEmpty(int row, int column);

    /**
     * Performs a bound check on the specified position.
     * @param position The position to check.
     * @return {@code true} if the specified position lies within the bounds of the board, {@code false} otherwise.
     */
    boolean checkBounds(IC4Position position);

    /**
     * Checks whether the specified position is "solid" while respecting the bounds of the board.
     * A position is solid if it occupied by a player, or if is one row below the bottom row (i.e., it is the "floor")
     * @param position The position to check.
     * @return {@code true} if the specified position is solid, {@code false} otherwise.
     * @throws IndexOutOfBoundsException If the specified position is outside the bounds of the board.
     */
    boolean isSolid(IC4Position position);

    /**
     * Checks whether the specified position is "solid" while respecting the bounds of the board.
     * A position is solid if it occupied by a player, or if is one row below the bottom row (i.e., it is the "floor")
     * @param row The row of the position to check.
     * @param column The column of the position to check.
     * @return {@code true} if the specified position is solid, {@code false} otherwise.
     * @throws IndexOutOfBoundsException If the specified position is outside the bounds of the board.
     */
    boolean isSolid(int row, int column);

    /**
     * Retrieves the player occupying the specified position, or {@link Optional#empty()} if the position is empty.
     * @param position The position to check.
     * @return The player occupying the specified position, or {@link Optional#empty()} if the position is empty.
     * @throws IndexOutOfBoundsException If the specified position is outside the bounds of the board.
     */
    Optional<IC4Player> getOccupyingPlayerOrDefault(IC4Position position);

    /**
     * Enumerates all empty positions of the board.
     * @return All empty positions of the board.
     */
    IC4Position[] getEmptyPositions();

    /**
     * Enumerates all positions occupied by the specified player.
     * @param player The player to check.
     * @return All positions occupied by the specified player.
     */
    IC4Position[] getPositionsByPlayer(IC4Player player);

    /**
     * Enumerates all legal positions of the board where a token can be placed.
     * @return All legal positions of the board.
     */
    IC4Position[] getLegalPositions();

    /**
     * Counts the number of empty positions on the board.
     * @return The number of empty positions on the board.
     */
    int countEmptyPositions();

    /**
     * Counts the number of positions occupied by the specified player.
     * @param player The player to check.
     * @return The number of positions occupied by the specified player.
     */
    int countPositionsByPlayer(IC4Player player);

    /**
     * Counts the number of legal positions on the board where a token can be placed.
     * @return The number of legal positions on the board.
     */
    int countLegalPositions();

    @Override
    IC4Board deepCopy();
}
