package de.fhdw.gaming.ipspiel23.c4.domain.impl.evaluation;

import java.util.Set;

import de.fhdw.gaming.ipspiel23.c4.domain.C4Direction;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4SolutionSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim;
import de.fhdw.gaming.ipspiel23.c4.domain.impl.C4SolutionSlim;

import static de.fhdw.gaming.ipspiel23.c4.domain.impl.C4BoardSlim.EMPTY_TOKEN;

/**
 * Represents a solution analyzer that scans the board in a specific direction.
 * <p>
 * Note: This is an internal API that may be subject to incompatible changes in future releases.
 * </p>
 */
public abstract class C4SolutionAnalyzer {
    
    /**
     * The direction in which the board is scanned.
     */
    private final C4Direction searchDirection;
    
    /**
     * The board that is scanned.
     */
    private final C4BoardSlim boardField;

    /**
     * A local copy of the board's row count for quicker access.
     */
    private final int rowMaxField;

    /**
     * A local copy of the board's column count for quicker access.
     */
    private final int colMaxField;

    /**
     * The internal number of tokens that need to be matched to form a solution.
     */
    private final int targetCountField;
    
    /**
     * Creates a new instance of {@link C4SolutionAnalyzer}.
     * @param board The board that is scanned.
     * @param searchDirection The direction in which the board is scanned.
     */
    protected C4SolutionAnalyzer(final C4BoardSlim board, final C4Direction searchDirection) {
        this.searchDirection = searchDirection;
        this.boardField = board;
        this.rowMaxField = board.getRowCount();
        this.colMaxField = board.getColumnCount();
        this.targetCountField = board.getMinimumSolutionSize() - 1;
    }
    
    /**
     * Lazily searches for the first solution on the board if the current solution is null.
     * @param currentSolution The current solution.
     * @param updateCache Whether to update the solution cache, preventing the same line to be checked again.
     * @return The first solution on the board or null if no solution was found.
     */
    public abstract IC4SolutionSlim tryFindFirstSolution(IC4SolutionSlim currentSolution, boolean updateCache);
    
    /**
     * Eagerly searches for all solutions on the board and adds them to the result set.
     * @param resultSet The result set to which the solutions are added, if any.
     * @param updateCache Whether to update the solution cache, preventing the same line to be checked again.
     */
    public abstract void findAllSolutions(Set<IC4SolutionSlim> resultSet, boolean updateCache);

    /**
     * The direction in which the board is scanned.
     */
    public C4Direction getSearchDirection() {
        return this.searchDirection;
    }
    
    /**
     * The board that is scanned.
     */
    protected C4BoardSlim board() {
        return this.boardField;
    }

    /**
     * Resets the internal cache.
     */
    public abstract void resetCache();
    
    /**
     * A local copy of the board's row count for quicker access.
     */
    protected int rowMax() {
        return this.rowMaxField;
    }
    
    /**
     * A local copy of the board's column count for quicker access.
     */
    protected int colMax() {
        return this.colMaxField;
    }
    
    /**
     * The internal number of tokens that need to be matched to form a solution.
     */
    protected int targetCount() {
        return this.targetCountField;
    }
    
    /**
     * Creates a new solution instance.
     * @param token The token that forms the solution.
     * @param matchEndRow The row index of the last token in the solution.
     * @param matchEndCol The column index of the last token in the solution.
     * @param size The number of tokens in the solution.
     * @return A new solution instance.
     */
    protected C4SolutionSlim solutionOf(final int token, final int matchEndRow, 
            final int matchEndCol, final int size) {
        return new C4SolutionSlim(boardField, token, matchEndRow, matchEndCol, this.searchDirection, size);
    }
    
    /**
     * Scans any additional tokens that belong to an already identified solution.
     * @param token The token that forms the solution.
     * @param row The row index of the last scanned token in the solution.
     * @param col The column index of the last scanned token in the solution.
     * @return The solution containing any additional tokens.
     */
    protected abstract C4SolutionSlim scanRemaining(int token, int row, int col);
    
    /**
     * Calculates the number of consecutive tokens with the same value without branching (prevent CPU pipeline stalls)
     * Returns count + 1 if the current token has the same value as the previous one AND 
     * the previous token was not 0, otherwise returns 0.
     * @param count The current number of consecutive tokens with the same value.
     * @param token The value of the current token.
     * @param previousToken The value of the previous token.
     * @return count + 1 if token == previousToken and previousToken != EMPTY_TOKEN else 0
     */
    protected static int countConsecutivesBranchless(final int count, final int token, final int previousToken) {
        // I hope this gets inlined by the JIT :)
        // this implements: return 0 + (token == previousToken && previousToken != EMPTY_TOKEN ? count + 1 : 0);
        // return 0 by default
        return 0
            // ... and add count + 1 ...
            + ((count + 1)
            // ... IFF the previous token has a value other than EMPTY_TOKEN ...
            & areNotEqualMask(previousToken, EMPTY_TOKEN)
            // ... AND the tokens are equal.
            & areEqualMask(token, previousToken));
    }

    /**
     * without branching, checks if {@code a} and {@code b} are not equal, returning the corresponding bit mask.
     * @param value1 the first value
     * @param value2 the other value
     * @return {@code -1 (TRUE)} iff a and b are not equal, {@code 0 (FALSE)} otherwise
     */
    protected static int areNotEqualMask(final int value1, final int value2) {
        return (-(value1 ^ value2)) >> 31;
    }

    /**
     * without branching, checks if {@code a} and {@code b} are equal, returning the corresponding bit mask.
     * @param value1 the first value
     * @param value2 the other value
     * @return {@code -1 (TRUE)} iff a and b are equal, {@code 0 (FALSE)} otherwise
     */
    protected static int areEqualMask(final int value1, final int value2) {
        return ~areNotEqualMask(value1, value2);
    }
}
