/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel23-Ssp.
 *
 * Ipspiel23-Ssp is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel23-Ssp is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel23-Ssp. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel23.ssp.domain.impl;

import de.fhdw.gaming.core.domain.DefaultGame;
import de.fhdw.gaming.core.domain.ObserverFactoryProvider;
import de.fhdw.gaming.ipspiel23.ssp.domain.SspGame;
import de.fhdw.gaming.ipspiel23.ssp.domain.SspMoveChecker;
import de.fhdw.gaming.ipspiel23.ssp.domain.SspPlayer;
import de.fhdw.gaming.ipspiel23.ssp.domain.SspState;
import de.fhdw.gaming.ipspiel23.ssp.domain.SspStrategy;
import de.fhdw.gaming.ipspiel23.ssp.moves.SspMove;
import de.fhdw.gaming.ipspiel23.ssp.moves.factory.SspMoveFactory;
import de.fhdw.gaming.ipspiel23.ssp.moves.impl.SspDefaultMoveFactory;

import java.util.Map;
import java.util.Optional;

/**
 * Implements the SSP game.
 */
final class SspGameImpl extends DefaultGame<SspPlayer, SspState, SspMove, SspStrategy> implements SspGame {

    /**
     * The move factory.
     */
    private final SspMoveFactory moveFactory;

    /**
     * Creates a Ssp game.
     *
     * @param id                        The ID of this game.
     * @param initialState              The initial state of the game.
     * @param strategies                The players' strategies.
     * @param maxComputationTimePerMove The maximum computation time per move in seconds.
     * @param moveChecker               The move checker.
     * @param observerFactoryProvider   The {@link ObserverFactoryProvider}.
     * @throws IllegalArgumentException if the player sets do not match.
     * @throws InterruptedException     if creating the game has been interrupted.
     */
    SspGameImpl(final int id, final SspState initialState, final Map<String, SspStrategy> strategies,
            final long maxComputationTimePerMove, final SspMoveChecker moveChecker,
            final ObserverFactoryProvider observerFactoryProvider)
            throws IllegalArgumentException, InterruptedException {

        super(id, initialState, strategies, maxComputationTimePerMove, moveChecker, observerFactoryProvider);
        this.moveFactory = new SspDefaultMoveFactory();
    }

    @Override
    public Optional<SspMove> chooseRandomMove(final SspPlayer player, final SspState state) {
        // Since all moves are equally good, just take stone...
        return Optional.of(this.moveFactory.createStoneMove());
    }

    @Override
    public String toString() {
        return String.format("SspGame[id=%d, %s]", this.getId(), this.gameToString());
    }
}
