package de.fhdw.gaming.ipspiel23.c4.strategies.internals;

import java.util.Optional;
import java.util.Random;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel23.c4.domain.C4PositionMaterializer;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Position;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4State;
import de.fhdw.gaming.ipspiel23.c4.moves.IC4Move;
import de.fhdw.gaming.ipspiel23.c4.moves.factory.IC4MoveFactory;
import de.fhdw.gaming.ipspiel23.c4.strategies.IC4Strategy;

/**
 * A strategy that picks a random legal move.
 */
public class C4RandomMoveStrategy implements IC4Strategy {

    /**
     * Your friendly neighborhood random number generator.
     */
    private static final Random RANDOM = new Random();

    /**
     * The move factory.
     */
    private final IC4MoveFactory moveFactory;

    /**
     * Creates a new random move strategy.
     * 
     * @param moveFactory the move factory
     */
    public C4RandomMoveStrategy(final IC4MoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    @Override
    public Optional<IC4Move> computeNextMove(final int gameId, final IC4Player player, final IC4State state)
            throws GameException, InterruptedException {
        // there can only be one move per column
        // we could also cache and re-use this array in ThreadLocal storage in the future
        final long[] legalPositions = new long[state.getBoard().getColumnCount()];
        // avoid temporary object allocation by directly using the internal board
        final int positionsWritten = state.getBoard()
            .getInternalBoard()
            .getLegalDematPositionsUnsafe(legalPositions);
        if (positionsWritten == 0) {
            return Optional.empty();
        }
        // pick a random legal position
        final int randomIndex = RANDOM.nextInt(positionsWritten);
        final long randomDematerializedPosition = legalPositions[randomIndex];
        // rematerialize only that position into a position object
        final IC4Position randomPosition = C4PositionMaterializer.rematerialize(randomDematerializedPosition);
        // create the move
        final IC4Move move = moveFactory.createMove(player, randomPosition);
        // done :)
        return Optional.of(move);
    }
    
    @Override
    public String toString() {
        return C4RandomMoveStrategy.class.getSimpleName();
    }
}
