package de.fhdw.gaming.ipspiel23.c4.domain.impl;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Board;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Player;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4Solution;
import de.fhdw.gaming.ipspiel23.c4.domain.IC4State;

/**
 * The default implementation of the {@link IC4State} interface.
 */
public class C4State implements IC4State {

    /**
     * The board of the Connect Four game.
     */
    private final IC4Board board;

    /**
     * The players of the Connect Four game, ordered by their respective tokens such that 
     * {@code index == players[index].getToken() - 1 }
     * holds true for all players.
     */
    private final IC4Player[] players;

    /**
     * The index of the current player in the {@link #players} array.
     */
    private int currentPlayerIndex;

    /**
     * The {@link Map} of players, mapping the player names to the respective player instances.
     */
    private Map<String, IC4Player> playerMap;

    /**
     * Constructs a new {@link C4State} instance.
     * @param board The board of the Connect Four game.
     * @param players The players of the Connect Four game, ordered by their respective tokens such that
     * {@code index == players[index].getToken() - 1 }
     * holds true for all players.
     * @throws NullPointerException If {@code board} or {@code players} is {@code null}.
     * @throws IllegalArgumentException If the {@code players} array is not ordered by player tokens ascending,
     * starting at token 1 and incrementing by 1 for each following player.
     */
    public C4State(final IC4Player[] players, final IC4Board board) {
        this.board = Objects.requireNonNull(board, "board");
        this.players = Objects.requireNonNull(players, "players");
        for (int i = 0; i < players.length; i++) {
            if (players[i].getToken() != i + 1) {
                throw new IllegalArgumentException(String.format("The 'players' array must be ordered by player "
                    + "tokens ascending, starting at token 1 and incrementing by 1 for each following player!. "
                    + "Player %s at index %s violates this rule!", 
                    players[i], i));
            }
        }
    }

    /**
     * Constructs a new {@link C4State} instance by copying the given {@code source} state.
     * @param source The source state to copy.
     */
    private C4State(final C4State source) {
        this.board = source.board.deepCopy();
        this.players = new IC4Player[source.players.length];
        this.currentPlayerIndex = source.currentPlayerIndex;
        for (int i = 0; i < this.players.length; i++) {
            this.players[i] = source.players[i].deepCopy();
        }
    }

    @Override
    public Map<String, IC4Player> getPlayers() {
        if (this.playerMap == null) {
            final HashMap<String, IC4Player> localPlayerMap = new HashMap<>(players.length);
            for (final IC4Player player : players) {
                localPlayerMap.put(player.getName(), player);
            }
            this.playerMap = localPlayerMap;
        }
        return this.playerMap;
    }

    @Override
    public Set<IC4Player> computeNextPlayers() {
        final IC4Player currentPlayer = this.getCurrentPlayer();
        if (currentPlayer.getState().equals(PlayerState.PLAYING)) {
            return Collections.singleton(currentPlayer);
        }
        return Collections.emptySet();
    }

    @Override
    public void nextTurn() {
        this.currentPlayerIndex = (this.currentPlayerIndex + 1) % this.players.length;
        //System.out.println(String.format("next player: %s", this.players[currentPlayerIndex]));
    }

    @Override
    public IC4State deepCopy() {
        // we implemented a mutable state pattern
        // *never* did any of the sparse documentation of the gaming framework state
        // anywhere that the state should be immutable
        // that would have been nice to know that ahead of time :P
        // Anyway, it should work with immutable states as well now.
        return new C4State(this);
    }

    @Override
    public IC4Board getBoard() {
        return this.board;
    }

    @Override
    public IC4Player getCurrentPlayer() {
        return this.players[currentPlayerIndex];
    }

    @Override
    public void onMoveCompleted() {
        // quick pass scanning for *any* solution
        final Optional<IC4Solution> solution = this.board.tryFindFirstSolution();
        if (solution.isPresent()) {
            // second slower pass scanning for *all* solutions.
            // we allow 2^31 - 1 different players on a board of up to 2^31 - 1 fields with 
            // a valid solution length between 2 and 2^31-1 ¯\_(ツ)_/¯
            // if we allow for stuff like this, we should probably also handle changed rules
            // where multiple winners could be possible :P
            final Set<IC4Solution> solutions = this.board.findAllSolutions();
            this.gameOver(Optional.of(solutions));
        } else {
            final boolean isFull = this.board.isFull();
            if (isFull) {
                this.gameOver(Optional.empty());
            }
        }
    }

    /**
     * Sets the state of all players to {@link PlayerState#LOST} except for the players that own at least 
     * one of the given {@code solutions}.
     * If no solutions are given, all players are set to {@link PlayerState#DRAW}.
     * @param optionalSolutions The solutions, if any.
     */
    private void gameOver(final Optional<Set<IC4Solution>> optionalSolutions) {
        if (optionalSolutions.isPresent()) {
            final Set<IC4Solution> solutions = optionalSolutions.get();
            for (final IC4Player player : this.players) {
                // we can't just do
                // sln.getOwner().setState(WON)
                // here because players are apparently deep copied with every move... :/
                // so yeah... Now we have this nightmare here:
                if (solutions.stream().anyMatch(sln -> sln.getOwner().equals(player))) {
                    player.setState(PlayerState.WON);
                } else {
                    player.setState(PlayerState.LOST);
                }
            }
        } else {
            for (final IC4Player player : players) {
                player.setState(PlayerState.DRAW);
            }
        }
    }

    @Override
    public String toString() {
        final StringBuilder builder = new StringBuilder(64);
        builder.append("C4State [board=").append(board.toString())
            .append(", players=").append(Arrays.toString(players))
            .append(", currentPlayer=").append(players[currentPlayerIndex].toString())
            .append(']');
        return builder.toString();
    }
}
