package de.fhdw.gaming.ipspiel23.dilemma.domain.internals;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel23.dilemma.domain.DilemmaAnswerType;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaGameBuilder;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaGameBuilderFactory;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayer;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaPlayerBuilder;
import de.fhdw.gaming.ipspiel23.dilemma.domain.IDilemmaStrategy;
import de.fhdw.gaming.ipspiel23.dilemma.moves.IDilemmaMoveFactory;
import de.fhdw.gaming.ipspiel23.dilemma.moves.internals.DilemmaDefaultMoveFactory;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.DilemmaDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.IDilemmaStrategyFactory;
import de.fhdw.gaming.ipspiel23.dilemma.strategy.IDilemmaStrategyFactoryProvider;

/**
 * Implements {@link GameBuilderFactory} by creating a Dilemma game builder.
 */
public final class DilemmaGameBuilderFactory implements IDilemmaGameBuilderFactory {

    /**
     * Number of players.
     */
    private static final int MIN_MAX_NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;
    /**
     * All available Dilemma strategies.
     */
    private final Set<IDilemmaStrategy> strategies;

    /**
     * Creates a Dilemma game factory. Dilemma strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public DilemmaGameBuilderFactory() {
        this(new DilemmaDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Dilemma game factory.
     *
     * @param strategyFactoryProvider The {@link IDilemmaStrategyFactoryProvider} for loading Dilemma strategies.
     */
    DilemmaGameBuilderFactory(final IDilemmaStrategyFactoryProvider strategyFactoryProvider) {
        final IDilemmaMoveFactory moveFactory = new DilemmaDefaultMoveFactory();

        final List<IDilemmaStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final IDilemmaStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Gefangenen-Dilemma";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return DilemmaGameBuilderFactory.MIN_MAX_NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return DilemmaGameBuilderFactory.MIN_MAX_NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public IDilemmaGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final IDilemmaGameBuilder gameBuilder = new DilemmaGameBuilder();

            @SuppressWarnings("unchecked")
            final Map<String, Object> gameData = inputProvider
                .needInteger(
                    GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                    "Maximum computation time per move in seconds ",
                    Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MinValueValidator<>(DilemmaGameBuilderFactory.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                    new MaxValueValidator<>(DilemmaGameBuilderFactory.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                .requestData("Game properties");

            /**
             * Changes the Computation time per Move according to the input of the player.
             */
            gameBuilder.changeMaximumComputationTimePerMove(
                (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1");
            final IDilemmaPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final IDilemmaStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2");
            final IDilemmaPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final IDilemmaStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Dilemma game was aborted: %s", e.getMessage()), e);
        }
    }


    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
            .needString(
                GameBuilderFactory.PARAM_PLAYER_NAME,
                "Name",
                Optional.empty(),
                new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
            .needInteger(
                IDilemmaGameBuilderFactory.PARAM_OUTCOME_ON_COOPERATE_COOPERATE,
                "Player's outcome on cooperate/cooperate",
                Optional.of(IDilemmaGameBuilder.DEFAULT_OUTCOME_COOPERATE_COOPERATE))
            .needInteger(
                IDilemmaGameBuilderFactory.PARAM_OUTCOME_ON_COOPERATE_DEFECT,
                "Player's outcome on cooperate/defect",
                Optional.of(IDilemmaGameBuilder.DEFAULT_OUTCOME_COOPERATE_DEFECT))
            .needInteger(
                IDilemmaGameBuilderFactory.PARAM_OUTCOME_ON_DEFECT_COOPERATE,
                "Player's outcome on defect/cooperate",
                Optional.of(IDilemmaGameBuilder.DEFAULT_OUTCOME_DEFECT_COOPERATE))
            .needInteger(
                IDilemmaGameBuilderFactory.PARAM_OUTCOME_ON_DEFECT_DEFECT,
                "Player's outcome on defect/defect",
                Optional.of(IDilemmaGameBuilder.DEFAULT_OUTCOME_DEFECT_DEFECT))
            .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Creates a Dilemma player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link IDilemmaPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private IDilemmaPlayer createPlayer(final IDilemmaPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<DilemmaAnswerType, Map<DilemmaAnswerType, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<DilemmaAnswerType, Double> possibleOutcomesNo = new LinkedHashMap<>();
        possibleOutcomesNo.put(
            DilemmaAnswerType.COOPERATE,
            (double) (Integer) playerData
                .get(IDilemmaGameBuilderFactory.PARAM_OUTCOME_ON_COOPERATE_COOPERATE));
        possibleOutcomesNo.put(
            DilemmaAnswerType.DEFECT,
            (double) (Integer) playerData
                .get(IDilemmaGameBuilderFactory.PARAM_OUTCOME_ON_COOPERATE_DEFECT));
        possibleOutcomes.put(DilemmaAnswerType.COOPERATE, possibleOutcomesNo);

        final Map<DilemmaAnswerType, Double> possibleOutcomesYes = new LinkedHashMap<>();
        possibleOutcomesYes.put(
            DilemmaAnswerType.COOPERATE,
            (double) (Integer) playerData
                .get(IDilemmaGameBuilderFactory.PARAM_OUTCOME_ON_DEFECT_COOPERATE));
        possibleOutcomesYes.put(
            DilemmaAnswerType.DEFECT,
            (double) (Integer) playerData
                .get(IDilemmaGameBuilderFactory.PARAM_OUTCOME_ON_DEFECT_DEFECT));
        possibleOutcomes.put(DilemmaAnswerType.DEFECT, possibleOutcomesYes);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
            .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a Dilemma strategy.
     *
     * @param playerData The requested player data.
     * @return The Dilemma strategy.
     */
    private IDilemmaStrategy getStrategy(final Map<String, Object> playerData) {
        return (IDilemmaStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
