/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel23-Ssp.
 *
 * Ipspiel23-Ssp is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel23-Ssp is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel23-Ssp. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel23.ssp.domain.impl;

import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.PlayerState;
import de.fhdw.gaming.ipspiel23.ssp.domain.SspPlayer;
import de.fhdw.gaming.ipspiel23.ssp.domain.SspState;
import de.fhdw.gaming.ipspiel23.ssp.domain.impl.outcomes.SspAnswer;


/**
 * Implements {@link SspState}.
 */
final class SspStateImpl implements SspState {

    /**
     * The first player.
     */
    private final SspPlayer firstPlayer;
    /**
     * The second player.
     */
    private final SspPlayer secondPlayer;

    /**
     * Creates a Ssp state.
     *
     * @param firstPlayer  The first player.
     * @param secondPlayer The second player.
     * @throws GameException if the state cannot be created according to the rules of the game.
     */
    SspStateImpl(final SspPlayer firstPlayer, final SspPlayer secondPlayer)
            throws GameException {

        this.firstPlayer = Objects.requireNonNull(firstPlayer, "firstPlayer");
        this.secondPlayer = Objects.requireNonNull(secondPlayer, "secondPlayer");

        if (this.firstPlayer.getName().equals(this.secondPlayer.getName())) {
            throw new IllegalArgumentException(
                    String.format("Both players have the same name '%s'.", this.firstPlayer.getName()));
        }
    }

    /**
     * Creates a Ssp state by copying an existing one.
     *
     * @param source The state to copy.
     */
    SspStateImpl(final SspStateImpl source) {
        this.firstPlayer = source.firstPlayer.deepCopy();
        this.secondPlayer = source.secondPlayer.deepCopy();
    }

    /**
     * Returns the first player.
     */
    @Override
    public SspPlayer getFirstPlayer() {
        return this.firstPlayer;
    }

    /**
     * Returns the second player.
     */
    @Override
    public SspPlayer getSecondPlayer() {
        return this.secondPlayer;
    }

    @Override
    public String toString() {
        return String.format("SspState[firstPlayer=%s, secondPlayer=%s]", this.firstPlayer, this.secondPlayer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof SspStateImpl) {
            final SspStateImpl other = (SspStateImpl) obj;
            return this.firstPlayer.equals(other.firstPlayer) && this.secondPlayer.equals(other.secondPlayer);
        }
        return false;
    }

    @Override
    public SspState deepCopy() {
        return new SspStateImpl(this);
    }

    @Override
    public int hashCode() {
        return Objects.hash(this.firstPlayer, this.secondPlayer);
    }

    @Override
    public Map<String, SspPlayer> getPlayers() {
        final Map<String, SspPlayer> result = new LinkedHashMap<>();
        result.put(this.firstPlayer.getName(), this.firstPlayer);
        result.put(this.secondPlayer.getName(), this.secondPlayer);
        return result;
    }

    @Override
    public Set<SspPlayer> computeNextPlayers() {
        final Set<SspPlayer> playersWithoutMove = new LinkedHashSet<>();
        if (this.firstPlayer.getAnswer().isEmpty()) {
            playersWithoutMove.add(this.firstPlayer);
        }
        if (this.secondPlayer.getAnswer().isEmpty()) {
            playersWithoutMove.add(this.secondPlayer);
        }
        return playersWithoutMove;
    }

    @Override
    public void nextTurn() {
        final Set<SspPlayer> playersWithoutMove = this.computeNextPlayers();
        if (playersWithoutMove.isEmpty()) {
            final SspAnswer answerOfFirstPlayer = this.firstPlayer.getAnswer().orElseThrow();
            final SspAnswer answerOfSecondPlayer = this.secondPlayer.getAnswer().orElseThrow();

            final Map<SspAnswer, Map<SspAnswer, Double>> outcomesPlayer1 = this.firstPlayer.getPossibleOutcomes();

            final Double outcomeOfFirstPlayer = outcomesPlayer1.get(answerOfFirstPlayer).get(answerOfSecondPlayer);
            this.firstPlayer.setState(outcomeToState(outcomeOfFirstPlayer));
            this.firstPlayer.setOutcome(outcomeOfFirstPlayer);

            // Same Map, but results are mirrored!
            final Map<SspAnswer, Map<SspAnswer, Double>> outcomesPlayer2 = this.secondPlayer.getPossibleOutcomes();

            final Double outcomeOfSecondPlayer = outcomesPlayer2.get(answerOfSecondPlayer)
                    .get(answerOfFirstPlayer);
            this.secondPlayer.setState(outcomeToState(outcomeOfSecondPlayer));
            this.secondPlayer.setOutcome(outcomeOfSecondPlayer);
            
        }
    }

    /**
     * Computes a player state from an outcome.
     *
     * @param outcome The player's outcome.
     */
    private static PlayerState outcomeToState(final Double outcome) {
        return outcome > 0.0 ? PlayerState.WON : outcome < 0.0 ? PlayerState.LOST : PlayerState.DRAW;
    }
}
