package de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.AbstractPlayer;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.domain.FzgPlayer;
import de.fhdw.gaming.ipspiel23.freizeitgestaltung.move.AnswerOptions;

/**
 * Implements {@link FzgPlayer}.
 * 
 */
public class FzgPlayerImpl extends AbstractPlayer<FzgPlayer> implements FzgPlayer {

    /**
     * answer of the Player.
     */
    private Optional<AnswerOptions> answer;

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<AnswerOptions, Map<AnswerOptions, Double>> possibleOutcomes;

    /**
     * Constructor.
     * 
     * @param name             Name of the Player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    protected FzgPlayerImpl(final String name, final Map<AnswerOptions, Map<AnswerOptions, Double>> possibleOutcomes) {
        super(name);
        this.possibleOutcomes = Collections
                .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
        this.answer = Optional.empty();
    }

    /**
     * Constructor for DeepCopy purpose.
     * 
     * @param source the player to copy.
     */
    public FzgPlayerImpl(final FzgPlayer source) {
        super(source);
        this.possibleOutcomes = source.getPossibleOutcomes();
        this.answer = source.getAnswer();
    }

    @Override
    public String toString() {
        return String
                .format("FzgPlayer[name=%s, state=%s, outcome=%s, answer=%s]",
                        this.getName(),
                        this.getState(),
                        this.getOutcome(),
                        this.answer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof FzgPlayerImpl) {
            final FzgPlayerImpl other = (FzgPlayerImpl) obj;
            return super.equals(obj) && this.answer.equals(other.answer)
                    && this.possibleOutcomes.equals(other.possibleOutcomes);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return super.hashCode() ^ Objects.hash(this.answer, this.possibleOutcomes);
    }

    @Override
    public FzgPlayer deepCopy() {
        return new FzgPlayerImpl(this);
    }

    @Override
    public void setAnswer(final AnswerOptions value) {
        if (this.answer.isPresent()) {
            throw new IllegalStateException(String.format("Player %s tried to change her answer.", this.getName()));
        }
        this.answer = Optional.of(value);
    }

    @Override
    public Map<AnswerOptions, Map<AnswerOptions, Double>> getPossibleOutcomes() {
        return possibleOutcomes;
    }

    @Override
    public Optional<AnswerOptions> getAnswer() {
        return this.answer;
    }
}
