/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-GD.
 *
 * ipspiel24-GD is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel24-GD is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel24-GD. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.GefangenenDilemma.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.GefangenenDilemma.domain.GDGameBuilder;
import de.fhdw.gaming.GefangenenDilemma.domain.GDGameBuilderFactory;
import de.fhdw.gaming.GefangenenDilemma.domain.GDPlayer;
import de.fhdw.gaming.GefangenenDilemma.domain.GDPlayerBuilder;
import de.fhdw.gaming.GefangenenDilemma.domain.GDStrategy;
import de.fhdw.gaming.GefangenenDilemma.domain.factory.GDDefaultStrategyFactoryProvider;
import de.fhdw.gaming.GefangenenDilemma.domain.factory.GDStrategyFactory;
import de.fhdw.gaming.GefangenenDilemma.domain.factory.GDStrategyFactoryProvider;
import de.fhdw.gaming.GefangenenDilemma.moves.factory.GDMoveFactory;
import de.fhdw.gaming.GefangenenDilemma.moves.impl.GDDefaultMoveFactory;
import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;

/**
 * Implements {@link GameBuilderFactory} by creating a Gefangenen-Dilemma game builder.
 */
public final class GDGameBuilderFactoryImpl implements GDGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * Implements {@link MOVES}.
     */
    public enum MOVES {
        /**
         * Implements{@link SNITCH_MOVE}.
         */
        SNITCH,
        /**
         * Implements{@link REMAIN_MOVE}.
         */
        REMAINSILENT
    }

    /**
     * All available Gefangenen-Dilemma strategies.
     */
    private final Set<GDStrategy> strategies;

    /**
     * Creates a Gefangenen-Dilemma game factory. 
     * Gefangenen-Dilemma strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public GDGameBuilderFactoryImpl() {
        this(new GDDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Gefangenen-Dilemma game factory.
     *
     * @param strategyFactoryProvider The {@link GDStrategyFactoryProvider} for loading Gefangenen-Dilemma strategies.
     */
    GDGameBuilderFactoryImpl(final GDStrategyFactoryProvider strategyFactoryProvider) {
        final GDMoveFactory moveFactory = new GDDefaultMoveFactory();

        final List<GDStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final GDStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "GefangenenDilemma";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return GDGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return GDGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public GDGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final GDGameBuilder gameBuilder = new GDGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(
                                    GDGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(
                                    GDGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1");
            final GDPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(),
                    firstPlayerData);
            final GDStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2");
            final GDPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(),
                    secondPlayerData);
            final GDStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(
                    String.format("Creating Gefangenen-Dilemma game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(
                        GameBuilderFactory.PARAM_PLAYER_NAME,
                        "Name",
                        Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(
                        GDGameBuilderFactory.OUTCOME_ON_SNITCH_SNITCH,
                        "Player's outcome on Snitch/Snitch",
                        Optional.of(-8))
                .needInteger(
                        GDGameBuilderFactory.OUTCOME_ON_SNITCH_REMAINSILENT,
                        "Player's outcome on Snitch/RemainSilent",
                        Optional.of(0))
                .needInteger(
                        GDGameBuilderFactory.OUTCOME_ON_REMAINSILENT_SNITCH,
                        "Player's outcome on RemainSilent/Snitch",
                        Optional.of(-10))
                .needInteger(
                        GDGameBuilderFactory.OUTCOME_ON_REMAINSILENT_REMAINSILENT,
                        "Player's outcome on RemainSilent/RemainSilent",
                        Optional.of(-1))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Creates a Gefangenen-Dilemma player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link GDPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private GDPlayer createPlayer(final GDPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {
        final Map<MOVES, Map<MOVES, Double>> possibleOutcomes = new LinkedHashMap<>();
        final Map<MOVES, Double> possibleOutcomesSnitch = new LinkedHashMap<>();
        possibleOutcomesSnitch.put(
                MOVES.SNITCH,
                (double) (Integer) playerData
                        .get(GDGameBuilderFactory.OUTCOME_ON_SNITCH_SNITCH));
        possibleOutcomesSnitch.put(
                MOVES.REMAINSILENT,
                (double) (Integer) playerData
                        .get(GDGameBuilderFactory.OUTCOME_ON_SNITCH_REMAINSILENT));
        possibleOutcomes.put(MOVES.SNITCH, possibleOutcomesSnitch);

        final Map<MOVES, Double> possibleOutcomesRemainSilent = new LinkedHashMap<>();
        possibleOutcomesRemainSilent.put(
                MOVES.SNITCH,
                (double) (Integer) playerData
                        .get(GDGameBuilderFactory.OUTCOME_ON_REMAINSILENT_SNITCH));
        possibleOutcomesRemainSilent.put(
                MOVES.REMAINSILENT,
                (double) (Integer) playerData
                        .get(GDGameBuilderFactory.OUTCOME_ON_REMAINSILENT_REMAINSILENT));
        possibleOutcomes.put(MOVES.REMAINSILENT, possibleOutcomesRemainSilent);
        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a Gefangenen-Dilemma strategy.
     *
     * @param playerData The requested player data.
     * @return The Gefangenen-Dilemma strategy.
     */
    private GDStrategy getStrategy(final Map<String, Object> playerData) {
        return (GDStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }

}
