/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-demo.
 *
 * Ipspiel24-demo is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel24-demo is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel24-demo. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.impl;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfPlayer;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfSide;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfPlayerBuilder;

/**
 * Implements {@link DemoPlayerBuilder}.
 */
final class MuenzwurfPlayerBuilderImpl implements MuenzwurfPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<MuenzwurfSide, Map<MuenzwurfSide, Double>>> possibleOutcomes;

    /**
     * Creates an {@link MuenzwurfPlayerBuilderImpl}.
     */
    MuenzwurfPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public MuenzwurfPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public MuenzwurfPlayerBuilder changePossibleOutcomes(final Map<MuenzwurfSide,
            Map<MuenzwurfSide, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public MuenzwurfPlayer build() throws GameException {
        return new MuenzwurfPlayerImpl(
                this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<MuenzwurfSide, Map<MuenzwurfSide, Double>> checkPossibleOutcomes(
            final Map<MuenzwurfSide, Map<MuenzwurfSide, Double>> outcomes) {
        this.checkPossibleOutcome(outcomes, MuenzwurfSide.TAILS, MuenzwurfSide.TAILS);
        this.checkPossibleOutcome(outcomes, MuenzwurfSide.TAILS, MuenzwurfSide.HEADS);
        this.checkPossibleOutcome(outcomes, MuenzwurfSide.HEADS, MuenzwurfSide.TAILS);
        this.checkPossibleOutcome(outcomes, MuenzwurfSide.HEADS, MuenzwurfSide.HEADS);
        return outcomes;
    }

    /**
     * Checks if a given outcome is defined for a player.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    private void checkPossibleOutcome(final Map<MuenzwurfSide,
            Map<MuenzwurfSide, Double>> outcomes, final MuenzwurfSide firstChoice,
            final MuenzwurfSide secondChoice) {
        if (outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                    String.format(
                            "No outcome defined for player '%s' and combination %s/%s.",
                            this.name,
                            toAnswer(firstChoice),
                            toAnswer(secondChoice)));
        }
    }

    /**
     * Maps a boolean value to a "yes" or "no" answer.
     *
     * @param value The value to be mapped.
     */
    private static String toAnswer(final MuenzwurfSide value) {
        return value == MuenzwurfSide.HEADS ? "Heads" : "Tails";
    }
}
