package de.fhdw.gaming.ipspiel24.fg.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.AbstractPlayer;
import de.fhdw.gaming.ipspiel24.fg.domain.FGActivity;
import de.fhdw.gaming.ipspiel24.fg.domain.FGPlayer;

/**
 * Implements {@link FGPlayer}.
 */
final class FGPlayerImpl extends AbstractPlayer<FGPlayer> implements FGPlayer {

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<FGActivity, Map<FGActivity, Double>> possibleOutcomes;
    /**
     * The answer of the player.
     */
    private Optional<FGActivity> answer;

    /**
     * Creates a FG player.
     *
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    FGPlayerImpl(final String name,
            final Map<FGActivity, Map<FGActivity, Double>> possibleOutcomes) {
        super(name);
        this.possibleOutcomes = Collections
                .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
        this.answer = Optional.empty();
    }

    /**
     * Creates a FG player.
     *
     * @param source The {@link FGPlayer} to copy.
     */
    FGPlayerImpl(final FGPlayer source) {
        super(source);
        this.possibleOutcomes = source.getPossibleOutcomes();
        this.answer = source.getAnswer();
    }

    @Override
    public String toString() {
        return String
                .format("FGPlayer[name=%s, state=%s, outcome=%s, answer=%s]", this.getName(), this.getState(),
                        this.getOutcome(),
                        this.answer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof FGPlayerImpl) {
            final FGPlayerImpl other = (FGPlayerImpl) obj;
            return super.equals(obj) && this.answer.equals(other.answer)
                    && this.possibleOutcomes.equals(other.possibleOutcomes);
        }
        return false;
    }

    @SuppressWarnings("PMD.UselessOverridingMethod")
    @Override
    public int hashCode() {
        return super.hashCode() ^ Objects.hash(this.answer, this.possibleOutcomes);
    }

    @Override
    public Map<FGActivity, Map<FGActivity, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public Optional<FGActivity> getAnswer() {
        return this.answer;
    }

    @Override
    public void setAnswer(final FGActivity newAnswer) {
        if (this.answer.isPresent()) {
            throw new IllegalStateException(String.format("Player %s tried to change her answer.", this.getName()));
        }
        this.answer = Optional.of(newAnswer);
    }

    @Override
    public FGPlayer deepCopy() {
        return new FGPlayerImpl(this);
    }
}
