/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-demo.
 *
 * Ipspiel24-demo is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel24-demo is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel24-demo. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.impl;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import de.fhdw.gaming.core.domain.AbstractPlayer;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfPlayer;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfSide;

/**
 * Implements {@link MuenzwurfPlayer}.
 */
final class MuenzwurfPlayerImpl extends AbstractPlayer<MuenzwurfPlayer> implements MuenzwurfPlayer {

    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private final Map<MuenzwurfSide, Map<MuenzwurfSide, Double>> possibleOutcomes;
    /**
     * The answer of the player.
     */
    private Optional<MuenzwurfSide> answer;
    
    /**
     * Decides whether the player gets points for heads/heads;tails/tails or
     *  heads/tails;tails/heads. True for the former.
     */
    private Boolean pointsForSame;

    /**
     * Creates a Muenzwurf player.
     *
     * @param name             The name of the player.
     * @param possibleOutcomes The possible outcomes of this player. The key for the first-level map is the answer of
     *                         the first player, the key for the second-level map is the answer of the second player.
     */
    MuenzwurfPlayerImpl(final String name,
            final Map<MuenzwurfSide, Map<MuenzwurfSide, Double>> possibleOutcomes) {
        super(name);
        this.possibleOutcomes = Collections
                .unmodifiableMap(new LinkedHashMap<>(Objects.requireNonNull(possibleOutcomes, "possibleOutcomes")));
        this.answer = Optional.empty();
    }

    /**
     * Creates a Muenzwurf player.
     *
     * @param source The {@link DemoPlayer} to copy.
     */
    MuenzwurfPlayerImpl(final MuenzwurfPlayer source) {
        super(source);
        this.possibleOutcomes = source.getPossibleOutcomes();
        this.answer = source.getSide();
    }
    
    /**
     * Gets pointsForSame.
     * @return The boolean currently save for pointsForSame.
     */
    @Override
    public Boolean getPointsForSame() {
        return this.pointsForSame;
    }
    
    /**
     * Sets pointsForSame.
     * @param toSet The boolean to set pointsForSame to.
     */
    @Override
    public void setPointsForSame(final Boolean toSet) {
        this.pointsForSame = toSet;
    }
    
    @Override
    public String toString() {
        return String
                .format("MuenzwurfPlayer[name=%s, state=%s, outcome=%s, answer=%s]", this.getName(), this.getState(),
                        this.getOutcome(),
                        this.answer);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj instanceof MuenzwurfPlayerImpl) {
            final MuenzwurfPlayerImpl other = (MuenzwurfPlayerImpl) obj;
            return super.equals(obj) && this.answer.equals(other.answer)
                    && this.possibleOutcomes.equals(other.possibleOutcomes);
        }
        return false;
    }

    @SuppressWarnings("PMD.UselessOverridingMethod")
    @Override
    public int hashCode() {
        return super.hashCode() ^ Objects.hash(this.answer, this.possibleOutcomes);
    }

    @Override
    public Map<MuenzwurfSide, Map<MuenzwurfSide, Double>> getPossibleOutcomes() {
        return this.possibleOutcomes;
    }

    @Override
    public Optional<MuenzwurfSide> getSide() {
        return this.answer;
    }

    @Override
    public void chooseSide(final MuenzwurfSide side) throws IllegalStateException {
        if (this.answer.isPresent()) {
            throw new IllegalStateException(String.format("Player %s tried to change her answer.", this.getName()));
        }
        this.answer = Optional.of(side);
    }

    @Override
    public MuenzwurfPlayer deepCopy() {
        return new MuenzwurfPlayerImpl(this);
    }
}
