/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-GD.
 *
 * ipspiel24-GD is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel24-GD is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel24-GD. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.GefangenenDilemma.strategy.stratsWithBrain;

import java.util.List;
import java.util.Map.Entry;
import java.util.Optional;

import de.fhdw.gaming.GefangenenDilemma.domain.GDPlayer;
import de.fhdw.gaming.GefangenenDilemma.domain.GDState;
import de.fhdw.gaming.GefangenenDilemma.domain.GDStrategy;
import de.fhdw.gaming.GefangenenDilemma.moves.GDMove;
import de.fhdw.gaming.GefangenenDilemma.moves.factory.GDMoveFactory;
import de.fhdw.gaming.GefangenenDilemma.moves.impl.GDRemainSilentMove;
import de.fhdw.gaming.GefangenenDilemma.moves.impl.GDSnitchMove;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Move;
import de.fhdw.gaming.memory.Key;
import de.fhdw.gaming.memory.impl.MemoryBrainImpl;
import de.fhdw.gaming.memory.impl.MemoryKeyImpl;
import de.fhdw.gaming.memory.impl.MemoryPairImpl;

/**
 * implement {@link GDStrategy} by creating a {@link GDPavlovStrategy}.
 */
public final class GDPavlovStrategy implements GDStrategy {

    /**
     * The factory for creating Gefangenen-Dilemma moves.
     */
    private final GDMoveFactory moveFactory;

    /**
     * Creates an {@link GDTitForTatStrategy}.
     *
     * @param moveFactory The factory for creating Gefangenen-Dilemma moves.
     */
    GDPavlovStrategy(final GDMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    @Override
    public Optional<GDMove> computeNextMove(
            int gameId,
            GDPlayer player,
            GDState state,
            long maxComputationTimePerMove)
            throws GameException, InterruptedException {

        GDPlayer opponent = null;
        for (Entry<String, GDPlayer> entry : state.getPlayers().entrySet()) {
            if (!entry.getValue().equals(player)) {
                opponent = entry.getValue();
            }
        }

        final Key key = new MemoryKeyImpl(player, opponent, this);

        List<MemoryPairImpl> check = null;
        if (MemoryBrainImpl.getInstance().getMemory(key) != null) {
            check = MemoryBrainImpl.getInstance().getMemory(key).show();
        }

        if (check == null || check.isEmpty()) {
            return Optional.of(this.moveFactory.createRemainSilentMove());
        }

        final int size = check.size();
        final Move<?, ?> lastMoveUsedOpponent = check.get(size - 1).getOpponentMove();
        final Move<?, ?> lastMoveUsedSelf = check.get(size - 1).getMyMove();

        return determineResponse(lastMoveUsedOpponent, lastMoveUsedSelf);
    }

    /**
     * Determines the response according to the last played Moves from the player and the opponent.
     * 
     * @param lastMoveUsedOpponent
     * @param lastMoveUsedSelf
     * @return
     */
    private Optional<GDMove> determineResponse(final Move<?, ?> lastMoveUsedOpponent,
            final Move<?, ?> lastMoveUsedSelf) {
        if (lastMoveUsedOpponent == null || lastMoveUsedSelf == null) {
            return Optional.of(this.moveFactory.createRemainSilentMove());
        }

        if (lastMoveUsedOpponent instanceof GDRemainSilentMove) {
            return Optional.of(repeatMove(lastMoveUsedSelf));
        } else if (lastMoveUsedOpponent instanceof GDSnitchMove) {
            return Optional.of(changeMove(lastMoveUsedSelf));
        } else {
            return Optional.of(this.moveFactory.createRemainSilentMove());
        }
    }

    /**
     * Private method to assist computeNextMove.
     * 
     * @param lastMove
     * @return new Strategy with same class as lastMove
     */
    private GDMove repeatMove(Move<?, ?> lastMove) {
        if (lastMove instanceof GDRemainSilentMove) {
            return this.moveFactory.createRemainSilentMove();
        } else if (lastMove instanceof GDSnitchMove) {
            return this.moveFactory.createSnitchMove();
        } else {
            return this.moveFactory.createRemainSilentMove();
        }
    }

    /**
     * Private method to assist computeNextMove.
     * 
     * @param lastMove
     * @return new Strategy with different class as lastMove
     */
    private GDMove changeMove(Move<?, ?> lastMove) {
        if (lastMove instanceof GDRemainSilentMove) {
            return this.moveFactory.createSnitchMove();
        } else if (lastMove instanceof GDSnitchMove) {
            return this.moveFactory.createRemainSilentMove();
        } else {
            return this.moveFactory.createRemainSilentMove();
        }
    }

    @Override
    public String toString() {
        return GDPavlovStrategy.class.getSimpleName();
    }

}
