/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-VierConnects-strategy-random.
 *
 * ipspiel24-VierConnects-strategy-random is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * ipspiel24-VierConnects-strategy-random is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * ipspiel24-VierConnects-strategy-random. If not, see <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.VierConnects.strategy.random;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Random;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsField;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsFieldState;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsPlayer;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsState;
import de.fhdw.gaming.ipspiel24.VierConnects.core.domain.VierConnectsStrategy;
import de.fhdw.gaming.ipspiel24.VierConnects.core.moves.VierConnectsMove;
import de.fhdw.gaming.ipspiel24.VierConnects.core.moves.factory.VierConnectsMoveFactory;

/**
 * Implements {@link VierConnectsStrategy} by randomly choosing the next possible move.
 */
public final class VierConnectsRandomMoveStrategy implements VierConnectsStrategy {

    /**
     * The random number generator.
     */
    private static final Random RANDOM = new Random();

    /**
     * The factory for creating VierConnects moves.
     */
    private final VierConnectsMoveFactory moveFactory;

    /**
     * Creates an {@link VierConnectsRandomMoveStrategy}.
     *
     * @param moveFactory The factory for creating VierConnects moves.
     */
    VierConnectsRandomMoveStrategy(final VierConnectsMoveFactory moveFactory) {
        this.moveFactory = moveFactory;
    }

    /**
     * Chooses some random move.
     */
    @Override
    public Optional<VierConnectsMove> computeNextMove(final int gameId, final VierConnectsPlayer player,
            final VierConnectsState state, final long maxComputationTimePerMove) throws GameException {

        final List<VierConnectsField> fields = new ArrayList<>(
                state.getBoard().getFieldsBeing(VierConnectsFieldState.EMPTY).values());

        if (fields.isEmpty()) {
            return Optional.empty();
        }
        final int index = RANDOM.nextInt(fields.size());
        final VierConnectsField field = fields.get(index);
        return Optional
                .of(this.moveFactory.createPlaceMarkMove(player.isUsingCrosses(), field.getPosition().getColumn()));
    }

    @Override
    public String toString() {
        return VierConnectsRandomMoveStrategy.class.getSimpleName();
    }
}
