
package de.fhdw.gaming.ipspiel24.fg.domain.impl;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;

import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.ipspiel24.fg.domain.FGActivity;
import de.fhdw.gaming.ipspiel24.fg.domain.FGPlayer;
import de.fhdw.gaming.ipspiel24.fg.domain.FGPlayerBuilder;

/**
 * Implements {@link FGPlayerBuilder}.
 */
final class FGPlayerBuilderImpl implements FGPlayerBuilder {

    /**
     * The name of the player.
     */
    private Optional<String> name;
    /**
     * The possible outcomes of this player. The key for the first-level map is the answer of the first player, the key
     * for the second-level map is the answer of the second player.
     */
    private Optional<Map<FGActivity, Map<FGActivity, Double>>> possibleOutcomes;

    /**
     * Creates an {@link FGPlayerBuilderImpl}.
     */
    FGPlayerBuilderImpl() {
        this.name = Optional.empty();
        this.possibleOutcomes = Optional.empty();
    }

    @Override
    public FGPlayerBuilderImpl changeName(final String newName) {
        this.name = Optional.of(newName);
        return this;
    }

    @Override
    public FGPlayerBuilder changePossibleOutcomes(final Map<FGActivity, Map<FGActivity, Double>> newPossibleOutcomes) {
        this.possibleOutcomes = Optional.of(newPossibleOutcomes);
        return this;
    }

    @Override
    public FGPlayer build() throws GameException {
        return new FGPlayerImpl(
                this.name.orElseThrow(),
                this.checkPossibleOutcomes(this.possibleOutcomes.orElseThrow()));
    }

    /**
     * Checks if all possible outcomes are defined for a player.
     *
     * @param outcomes The possible outcomes for the player.
     */
    private Map<FGActivity, Map<FGActivity, Double>> checkPossibleOutcomes(
            final Map<FGActivity, Map<FGActivity, Double>> outcomes) {
        this.checkPossibleOutcome(outcomes, FGActivity.FOOTBALL, FGActivity.FOOTBALL);
        this.checkPossibleOutcome(outcomes, FGActivity.FOOTBALL, FGActivity.CINEMA);
        this.checkPossibleOutcome(outcomes, FGActivity.CINEMA, FGActivity.FOOTBALL);
        this.checkPossibleOutcome(outcomes, FGActivity.CINEMA, FGActivity.CINEMA);
        return outcomes;
    }

    /**
     * Checks if a given outcome is defined for a player.
     *
     * @param outcomes     The possible outcomes for the player.
     * @param firstChoice  The choice of the first player.
     * @param secondChoice The choice of the second player.
     */
    private void checkPossibleOutcome(final Map<FGActivity, Map<FGActivity, Double>> outcomes,
            final FGActivity firstChoice,
            final FGActivity secondChoice) {
        if (outcomes.getOrDefault(firstChoice, Collections.emptyMap()).get(secondChoice) == null) {
            throw new IllegalArgumentException(
                    String.format(
                            "No outcome defined for player '%s' and combination %s/%s.",
                            this.name,
                            toAnswer(firstChoice),
                            toAnswer(secondChoice)));
        }
    }

    /**
     * Maps a FGActivity value to a "yes" or "no" answer.
     *
     * @param value The value to be mapped.
     */
    private static String toAnswer(final FGActivity value) {
        return value.equals(FGActivity.CINEMA) ? "cinema" : "football";
    }
}
