
package de.fhdw.gaming.ipspiel24.fg.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel24.fg.domain.FGActivity;
import de.fhdw.gaming.ipspiel24.fg.domain.FGGameBuilder;
import de.fhdw.gaming.ipspiel24.fg.domain.FGGameBuilderFactory;
import de.fhdw.gaming.ipspiel24.fg.domain.FGPlayer;
import de.fhdw.gaming.ipspiel24.fg.domain.FGPlayerBuilder;
import de.fhdw.gaming.ipspiel24.fg.domain.FGStrategy;
import de.fhdw.gaming.ipspiel24.fg.domain.factory.FGDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel24.fg.domain.factory.FGStrategyFactory;
import de.fhdw.gaming.ipspiel24.fg.domain.factory.FGStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel24.fg.moves.factory.FGMoveFactory;
import de.fhdw.gaming.ipspiel24.fg.moves.impl.FGDefaultMoveFactory;

/**
 * Implements {@link GameBuilderFactory} by creating a FG game builder.
 */
public final class FGGameBuilderFactoryImpl implements FGGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available FG strategies.
     */
    private final Set<FGStrategy> strategies;

    /**
     * Creates a FG game factory. FG strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public FGGameBuilderFactoryImpl() {
        this(new FGDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a FG game factory.
     *
     * @param strategyFactoryProvider The {@link FGStrategyFactoryProvider} for loading FG strategies.
     */
    FGGameBuilderFactoryImpl(final FGStrategyFactoryProvider strategyFactoryProvider) {
        final FGMoveFactory moveFactory = new FGDefaultMoveFactory();

        final List<FGStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final FGStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "FG";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return FGGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return FGGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public FGGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final FGGameBuilder gameBuilder = new FGGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(FGGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(FGGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider,
                    "Player 1", true);
            final FGPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final FGStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider,
                    "Player 2", false);
            final FGPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final FGStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating FG game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @param isFirstPlayer Differentiate between Player 1 and 2 payout
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider,
            final String title, final boolean isFirstPlayer)
            throws GameException, InputProviderException {
            inputProvider
            .needString(
                    GameBuilderFactory.PARAM_PLAYER_NAME,
                    isFirstPlayer ? "Klaus" : "Gabi",
                    Optional.empty(),
                    new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
            .needInteger(
                    FGGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_FOOTBALL_FOOTBALL,
                    "Player's outcome on Football/Football",
                    isFirstPlayer ? Optional.of(2) : Optional.of(1))
            .needInteger(
                    FGGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_FOOTBALL_CINEMA,
                    "Player's outcome on Football/Cinema",
                    Optional.of(0))
            .needInteger(
                    FGGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_FOOTBALL,
                    "Player's outcome on Cinema/Football",
                    Optional.of(0))
            .needInteger(
                    FGGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_CINEMA,
                    "Player's outcome on Cinema/Cinema",
                    isFirstPlayer ? Optional.of(1) : Optional.of(2))
            .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);



        return inputProvider.requestData(title);
    }

    /**
     * Creates a FG player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link FGPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private FGPlayer createPlayer(final FGPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<FGActivity, Map<FGActivity, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<FGActivity, Double> possibleOutcomesNo = new LinkedHashMap<>();
        possibleOutcomesNo.put(
                FGActivity.FOOTBALL,
                (double) (Integer) playerData.get(FGGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_FOOTBALL_FOOTBALL));
        possibleOutcomesNo.put(
                FGActivity.CINEMA,
                (double) (Integer) playerData.get(FGGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_FOOTBALL_CINEMA));
        possibleOutcomes.put(FGActivity.FOOTBALL, possibleOutcomesNo);

        final Map<FGActivity, Double> possibleOutcomesYes = new LinkedHashMap<>();
        possibleOutcomesYes.put(
                FGActivity.FOOTBALL,
                (double) (Integer) playerData.get(FGGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_FOOTBALL));
        possibleOutcomesYes.put(
                FGActivity.CINEMA,
                (double) (Integer) playerData.get(FGGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_CINEMA_CINEMA));
        possibleOutcomes.put(FGActivity.CINEMA, possibleOutcomesYes);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a FG strategy.
     *
     * @param playerData The requested player data.
     * @return The FG strategy.
     */
    private FGStrategy getStrategy(final Map<String, Object> playerData) {
        return (FGStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
