package de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfGameBuilder;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfGameBuilderFactory;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfPlayer;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfPlayerBuilder;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfStrategy;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.MuenzwurfSide;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.factory.MuenzwurfDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.factory.MuenzwurfStrategyFactory;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.domain.factory.MuenzwurfStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.moves.factory.MuenzwurfMoveFactory;
import de.fhdw.gaming.ipspiel24.muenzwurf.core.moves.impl.MuenzwurfDefaultMoveFactory;

/**
 * Implements {@link GameBuilderFactory} by creating a Muenzwurf game builder.
 */
public final class MuenzwurfGameBuilderFactoryImpl implements MuenzwurfGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Muenzwurf strategies.
     */
    private final Set<MuenzwurfStrategy> strategies;

    /**
     * Creates a Muenzwurf game factory. Muenzwurf strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public MuenzwurfGameBuilderFactoryImpl() {
        this(new MuenzwurfDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Muenzwurf game factory.
     *
     * @param strategyFactoryProvider The {@link MuenzwurfStrategyFactoryProvider} for loading Muenzwurf strategies.
     */
    MuenzwurfGameBuilderFactoryImpl(final MuenzwurfStrategyFactoryProvider strategyFactoryProvider) {
        final MuenzwurfMoveFactory moveFactory = new MuenzwurfDefaultMoveFactory();

        final List<MuenzwurfStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final MuenzwurfStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Muenzwurf";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return MuenzwurfGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return MuenzwurfGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public MuenzwurfGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final MuenzwurfGameBuilder gameBuilder = new MuenzwurfGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(MuenzwurfGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(MuenzwurfGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider,
                    "Player 1", Optional.empty());
            final MuenzwurfPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final MuenzwurfStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider,
                    "Player 2", Optional.of(
                    !(Boolean) firstPlayerData.get(MuenzwurfGameBuilderFactory.PARAM_PLAYER_POINTGAIN_ON_SAME_RESULT)));
            final MuenzwurfPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final MuenzwurfStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Muenzwurf game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @param pointsOnSame Whether the player gains poins on the same chosen side.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title,
            final Optional<Boolean> pointsOnSame)
            throws GameException, InputProviderException {

        inputProvider
                .needString(
                        GameBuilderFactory.PARAM_PLAYER_NAME,
                        "Name",
                        Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needBoolean(
                        MuenzwurfGameBuilderFactory.PARAM_PLAYER_POINTGAIN_ON_SAME_RESULT,
                        "Player gets points on same result",
                        Optional.of(Boolean.TRUE))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);
        
        if (pointsOnSame.isPresent()) {
            inputProvider
                    .fixedBoolean(MuenzwurfGameBuilderFactory.PARAM_PLAYER_POINTGAIN_ON_SAME_RESULT, 
                            pointsOnSame.get());
        }

        return inputProvider.requestData(title);
    }

    /**
     * Creates a Muenzwurf player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link MuenzwurfPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private MuenzwurfPlayer createPlayer(final MuenzwurfPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<MuenzwurfSide, Map<MuenzwurfSide, Double>> possibleOutcomes = new LinkedHashMap<>();
        final Map<MuenzwurfSide, Double> possibleOutcomesTails = new LinkedHashMap<>();
        final Map<MuenzwurfSide, Double> possibleOutcomesHeads = new LinkedHashMap<>();
        final Map<MuenzwurfSide, Double> possibleOutcomesEdge = new LinkedHashMap<>();
        if ((Boolean) playerData.get(MuenzwurfGameBuilderFactory.PARAM_PLAYER_POINTGAIN_ON_SAME_RESULT)) { 
            possibleOutcomesTails.put(
                    MuenzwurfSide.TAILS,
                    1.0);
            possibleOutcomesTails.put(
                    MuenzwurfSide.HEADS,
                    -1.0);
            possibleOutcomesTails.put(
                    MuenzwurfSide.EDGE,
                    -1.0);
            possibleOutcomes.put(MuenzwurfSide.TAILS, possibleOutcomesTails);
    
            possibleOutcomesHeads.put(
                    MuenzwurfSide.TAILS,
                    -1.0);
            possibleOutcomesHeads.put(
                    MuenzwurfSide.HEADS,
                    1.0);
            possibleOutcomesHeads.put(
                    MuenzwurfSide.EDGE,
                    -1.0);
            possibleOutcomes.put(MuenzwurfSide.HEADS, possibleOutcomesHeads);
            
            possibleOutcomesEdge.put(
                    MuenzwurfSide.TAILS,
                    -1.0);
            possibleOutcomesEdge.put(
                    MuenzwurfSide.HEADS,
                    -1.0);
            possibleOutcomesEdge.put(
                    MuenzwurfSide.EDGE,
                    1.0);
            possibleOutcomes.put(MuenzwurfSide.EDGE, possibleOutcomesEdge);
        } else {
            possibleOutcomesTails.put(
                    MuenzwurfSide.TAILS,
                    -1.0);
            possibleOutcomesTails.put(
                    MuenzwurfSide.HEADS,
                    1.0);
            possibleOutcomesTails.put(
                    MuenzwurfSide.EDGE,
                    1.0);
            possibleOutcomes.put(MuenzwurfSide.TAILS, possibleOutcomesTails);

            possibleOutcomesHeads.put(
                    MuenzwurfSide.TAILS,
                    1.0);
            possibleOutcomesHeads.put(
                    MuenzwurfSide.HEADS,
                    -1.0);
            possibleOutcomesHeads.put(
                    MuenzwurfSide.EDGE,
                    1.0);
            possibleOutcomes.put(MuenzwurfSide.HEADS, possibleOutcomesHeads);
            
            possibleOutcomesEdge.put(
                    MuenzwurfSide.TAILS,
                    1.0);
            possibleOutcomesEdge.put(
                    MuenzwurfSide.HEADS,
                    1.0);
            possibleOutcomesEdge.put(
                    MuenzwurfSide.EDGE,
                    -1.0);
            possibleOutcomes.put(MuenzwurfSide.EDGE, possibleOutcomesEdge);
        }

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a Muenzwurf strategy.
     *
     * @param playerData The requested player data.
     * @return The Muenzwurf strategy.
     */
    private MuenzwurfStrategy getStrategy(final Map<String, Object> playerData) {
        return (MuenzwurfStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
