/*
 * Copyright © 2021-2023 Fachhochschule für die Wirtschaft (FHDW) Hannover
 *
 * This file is part of ipspiel24-Ssp.
 *
 * Ipspiel24-Ssp is free software: you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later
 * version.
 *
 * Ipspiel24-Ssp is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with ipspiel24-Ssp. If not, see
 * <http://www.gnu.org/licenses/>.
 */
package de.fhdw.gaming.ipspiel24.ssp.domain.impl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import de.fhdw.gaming.core.domain.GameBuilder;
import de.fhdw.gaming.core.domain.GameBuilderFactory;
import de.fhdw.gaming.core.domain.GameException;
import de.fhdw.gaming.core.domain.Strategy;
import de.fhdw.gaming.core.ui.InputProvider;
import de.fhdw.gaming.core.ui.InputProviderException;
import de.fhdw.gaming.core.ui.type.validator.MaxValueValidator;
import de.fhdw.gaming.core.ui.type.validator.MinValueValidator;
import de.fhdw.gaming.core.ui.type.validator.PatternValidator;
import de.fhdw.gaming.ipspiel24.ssp.domain.SspGameBuilder;
import de.fhdw.gaming.ipspiel24.ssp.domain.SspGameBuilderFactory;
import de.fhdw.gaming.ipspiel24.ssp.domain.SspPlayer;
import de.fhdw.gaming.ipspiel24.ssp.domain.SspPlayerBuilder;
import de.fhdw.gaming.ipspiel24.ssp.domain.SspStrategy;
import de.fhdw.gaming.ipspiel24.ssp.domain.factory.SspDefaultStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel24.ssp.domain.factory.SspStrategyFactory;
import de.fhdw.gaming.ipspiel24.ssp.domain.factory.SspStrategyFactoryProvider;
import de.fhdw.gaming.ipspiel24.ssp.moves.factory.SspMoveFactory;
import de.fhdw.gaming.ipspiel24.ssp.moves.impl.SspDefaultMoveFactory;
import de.fhdw.gaming.ipspiel24.ssp.domain.SspPossibleChoice;

/**
 * Implements {@link GameBuilderFactory} by creating a Ssp game builder.
 */
public final class SspGameBuilderFactoryImpl implements SspGameBuilderFactory {

    /**
     * The number of players.
     */
    private static final int NUMBER_OF_PLAYERS = 2;
    /**
     * Smallest allowed maximum computation time per move in seconds.
     */
    private static final int MIN_MAX_COMPUTATION_TIME_PER_MOVE = 1;
    /**
     * Largest allowed maximum computation time per move in seconds.
     */
    private static final int MAX_MAX_COMPUTATION_TIME_PER_MOVE = 3600;

    /**
     * All available Ssp strategies.
     */
    private final Set<SspStrategy> strategies;

    /**
     * Creates a Ssp game factory. Ssp strategies are loaded by using the {@link java.util.ServiceLoader}.
     * <p>
     * This constructor is meant to be used by the {@link java.util.ServiceLoader}.
     */
    public SspGameBuilderFactoryImpl() {
        this(new SspDefaultStrategyFactoryProvider());
    }

    /**
     * Creates a Ssp game factory.
     *
     * @param strategyFactoryProvider The {@link SspStrategyFactoryProvider} for loading Ssp strategies.
     */
    SspGameBuilderFactoryImpl(final SspStrategyFactoryProvider strategyFactoryProvider) {
        final SspMoveFactory moveFactory = new SspDefaultMoveFactory();

        final List<SspStrategyFactory> factories = strategyFactoryProvider.getStrategyFactories();
        this.strategies = new LinkedHashSet<>();
        for (final SspStrategyFactory factory : factories) {
            this.strategies.add(factory.create(moveFactory));
        }
    }

    @Override
    public String getName() {
        return "Ssp";
    }

    @Override
    public int getMinimumNumberOfPlayers() {
        return SspGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public int getMaximumNumberOfPlayers() {
        return SspGameBuilderFactoryImpl.NUMBER_OF_PLAYERS;
    }

    @Override
    public List<? extends Strategy<?, ?, ?>> getStrategies() {
        return new ArrayList<>(this.strategies);
    }

    @Override
    public SspGameBuilder createGameBuilder(final InputProvider inputProvider) throws GameException {
        try {
            final SspGameBuilder gameBuilder = new SspGameBuilderImpl();

            @SuppressWarnings("unchecked")
            final Map<String,
                    Object> gameData = inputProvider.needInteger(
                            GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE,
                            "Maximum computation time per move in seconds",
                            Optional.of(GameBuilder.DEFAULT_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MinValueValidator<>(SspGameBuilderFactoryImpl.MIN_MAX_COMPUTATION_TIME_PER_MOVE),
                            new MaxValueValidator<>(SspGameBuilderFactoryImpl.MAX_MAX_COMPUTATION_TIME_PER_MOVE))
                            .requestData("Game properties");

            gameBuilder.changeMaximumComputationTimePerMove(
                    (Integer) gameData.get(GameBuilderFactory.PARAM_MAX_COMPUTATION_TIME_PER_MOVE));

            final InputProvider firstPlayerInputProvider = inputProvider.getNext(gameData);
            final Map<String, Object> firstPlayerData = this.requestPlayerData(firstPlayerInputProvider, "Player 1");
            final SspPlayer firstPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), firstPlayerData);
            final SspStrategy firstPlayerStrategy = this.getStrategy(firstPlayerData);
            gameBuilder.addPlayer(firstPlayer, firstPlayerStrategy);

            final InputProvider secondPlayerInputProvider = firstPlayerInputProvider.getNext(firstPlayerData);
            final Map<String, Object> secondPlayerData = this.requestPlayerData(secondPlayerInputProvider, "Player 2");
            final SspPlayer secondPlayer = this.createPlayer(gameBuilder.createPlayerBuilder(), secondPlayerData);
            final SspStrategy secondPlayerStrategy = this.getStrategy(secondPlayerData);
            gameBuilder.addPlayer(secondPlayer, secondPlayerStrategy);

            return gameBuilder;
        } catch (final InputProviderException e) {
            throw new GameException(String.format("Creating Ssp game was aborted: %s", e.getMessage()), e);
        }
    }

    /**
     * Returns data for a player builder.
     *
     * @param inputProvider The input provider.
     * @param title         The title for the UI.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    @SuppressWarnings("unchecked")
    private Map<String, Object> requestPlayerData(final InputProvider inputProvider, final String title)
            throws GameException, InputProviderException {

        inputProvider
                .needString(
                        GameBuilderFactory.PARAM_PLAYER_NAME,
                        "Name",
                        Optional.empty(),
                        new PatternValidator(Pattern.compile("\\S+(\\s+\\S+)*")))
                .needInteger(
                        SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ROCK_ROCK,
                        "Player's outcome on Rock/Rock",
                        Optional.of(0))
                .needInteger(
                        SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ROCK_PAPER,
                        "Player's outcome on Rock/Paper",
                        Optional.of(-1))
                .needInteger(
                        SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ROCK_SCISSORS,
                        "Player's outcome on Rock/Scissors",
                        Optional.of(1))
                .needInteger(
                        SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPER_ROCK,
                        "Player's outcome on Paper/Rock",
                        Optional.of(1))
                .needInteger(
                        SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPER_PAPER,
                        "Player's outcome on Paper/Paper",
                        Optional.of(0))
                .needInteger(
                        SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPER_SCISSORS,
                        "Player's outcome on Paper/Scissors",
                        Optional.of(-1))
                .needInteger(
                        SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCISSORS_ROCK,
                        "Player's outcome on Scissors/Rock",
                        Optional.of(-1))
                .needInteger(
                        SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCISSORS_PAPER,
                        "Player's outcome on Scissors/Paper",
                        Optional.of(1))
                .needInteger(
                        SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SC_SC,
                        "Player's outcome on Scissors/Scissors",
                        Optional.of(0))
                .needObject(GameBuilderFactory.PARAM_PLAYER_STRATEGY, "Strategy", Optional.empty(), this.strategies);

        return inputProvider.requestData(title);
    }

    /**
     * Creates a Ssp player.
     *
     * @param playerBuilder The player builder.
     * @param playerData    The requested player data.
     * @return The created {@link SspPlayer}.
     * @throws InputProviderException if the operation has been aborted prematurely (e.g. if the user cancelled a
     *                                dialog).
     */
    private SspPlayer createPlayer(final SspPlayerBuilder playerBuilder,
            final Map<String, Object> playerData) throws GameException, InputProviderException {

        final Map<SspPossibleChoice, Map<SspPossibleChoice, Double>> possibleOutcomes = new LinkedHashMap<>();

        final Map<SspPossibleChoice, Double> possibleOutcomesRock = new LinkedHashMap<>();
        possibleOutcomesRock.put(
                SspPossibleChoice.SPC_ROCK,
                (double) (Integer) playerData.get(SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ROCK_ROCK));
        possibleOutcomesRock.put(
                SspPossibleChoice.SPC_PAPER,
                (double) (Integer) playerData.get(SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ROCK_PAPER));
        possibleOutcomesRock.put(
                SspPossibleChoice.SPC_SCISSORS,
                (double) (Integer) playerData.get(SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_ROCK_SCISSORS));
        possibleOutcomes.put(SspPossibleChoice.SPC_ROCK, possibleOutcomesRock);

        final Map<SspPossibleChoice, Double> possibleOutcomesPaper = new LinkedHashMap<>();
        possibleOutcomesPaper.put(
                SspPossibleChoice.SPC_ROCK,
                (double) (Integer) playerData.get(SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPER_ROCK));
        possibleOutcomesPaper.put(
                SspPossibleChoice.SPC_PAPER,
                (double) (Integer) playerData.get(SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPER_PAPER));
        possibleOutcomesPaper.put(
                SspPossibleChoice.SPC_SCISSORS,
                (double) (Integer) playerData.get(SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_PAPER_SCISSORS));
        possibleOutcomes.put(SspPossibleChoice.SPC_PAPER, possibleOutcomesPaper);

        final Map<SspPossibleChoice, Double> possibleOutcomesScissors = new LinkedHashMap<>();
        possibleOutcomesScissors.put(
                SspPossibleChoice.SPC_ROCK,
                (double) (Integer) playerData.get(SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCISSORS_ROCK));
        possibleOutcomesScissors.put(
                SspPossibleChoice.SPC_PAPER,
                (double) (Integer) playerData.get(SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SCISSORS_PAPER));
        possibleOutcomesScissors.put(
                SspPossibleChoice.SPC_SCISSORS,
                (double) (Integer) playerData.get(SspGameBuilderFactory.PARAM_PLAYER_OUTCOME_ON_SC_SC));
        possibleOutcomes.put(SspPossibleChoice.SPC_SCISSORS, possibleOutcomesScissors);

        return playerBuilder.changeName((String) playerData.get(GameBuilderFactory.PARAM_PLAYER_NAME))
                .changePossibleOutcomes(possibleOutcomes).build();
    }

    /**
     * Returns a Ssp strategy.
     *
     * @param playerData The requested player data.
     * @return The Ssp strategy.
     */
    private SspStrategy getStrategy(final Map<String, Object> playerData) {
        return (SspStrategy) playerData.get(GameBuilderFactory.PARAM_PLAYER_STRATEGY);
    }
}
